"""
Module STEPModel
"""

__version__ = "$Revision$"
__revision__ = "$Id$"

import sys, os, datetime

import CSEPGeneric, CSEPFile, CSEPInputParams
from Forecast import Forecast
from OneDayForecast import OneDayForecast
from CSEPLogging import CSEPLogging


#-------------------------------------------------------------------------------------
#
# STEP forecast model.
#
# This class is designed to invoke a one-day STEP forecast model for New Zealand
# natural laboratory.
# It prepares input catalog data, formats input file with model parameters, 
# and invokes the STEP model. It places forecast file under user specified 
# directory.
#
class STEPModel (OneDayForecast):

    # Static data of the class
    
    # Keyword identifying type of the class
    Type = "STEP"
    
    # Name of the rateGrid file used for plotting
    __GridRateFile = "rateGrid_gmt.txt"
    
    # Name of the RELM_poly file
    __RELMPolyFile = "NZTestArea.txt"
    
    # Center code path for the model
    __modelPath = os.path.join(os.environ['NZCENTERCODE'],
                               'STEP')
    
    # Option to specify recursion limit in Matlab
    __recursionLimitOption = 'matlabRecursionLimit'
    
    # This data is static for the class - safe since we generate only one 
    # forecast per model for any CSEP run.
    __defaultArgs = {__recursionLimitOption : '800'}
    
    
    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. Default is None.
    # 
    def __init__ (self, dir_path, args=None):
        """ Initialization for STEPModel class"""
        
        # Use later than actual download start date for the input catalog
        # to make STEP code happy
        OneDayForecast.__init__(self, dir_path)

        # Input arguments for the model were provided:
        self.__args = CSEPInputParams.CSEPInputParams.parse(STEPModel.__defaultArgs, 
                                                            args)
       

    #--------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""
        
        return self.Type

    #--------------------------------------------------------------------
    #
    # Return full path for the result forecast file.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the filename.
    #
    def filename (self):
        """ Returns filename of generated forecast."""

        # Model is generating Matlab format forecast, change extension for the
        # result filename
        return CSEPFile.Name.matlab(Forecast.filename(self))
     

    #--------------------------------------------------------------------
    #
    # Return filename for the input catalog data. It overwrites parent's
    # method since STEP is using Matlab format of input catalog.
    #
    # Input: None.
    #
    # Output:
    #        String identifying the filename.
    #
    def inputCatalogFilename (self):
        """ Returns filename of input catalog data."""
        
        return CSEPFile.Name.matlab(Forecast.inputCatalogFilename(self))

        
    #--------------------------------------------------------------------
    #
    # Write input parameter file for the model.
    #
    # Input: None.
    #        
    def writeParameterFile (self):
        """ Format input parameter file for the model.
            Created file will be used by Matlab script that invokes the
            model."""

        fhandle = Forecast.writeParameterFile(self)
        
        # Start time of the data - to make STEP code happy use later than
        # 1985/1/1 - 1992/1/1
        line = "1 1 1992 0 0 0\n"
        fhandle.write(line)
        
        # Test date - append start "hour minute second" as all zero's 
        line = "%s %s %s 0 0 0\n" \
               %(self.start_date.day, self.start_date.month, self.start_date.year)
        fhandle.write(line)
        
        # Duration in days
        line = "1\n"
        fhandle.write(line)
        
        # Path to the input catalog file
        fhandle.write(os.path.join(self.catalogDir, self.inputCatalogFilename()) + "\n")
        
        # Path to the output forecast file
        fhandle.write(self.filename() + "\n")
        
        # Path to the rateGrid file - for plotting
        fhandle.write(os.path.join(self.catalogDir, self.__GridRateFile) + "\n")

        # Path to the RELM_poly.txt file
        fhandle.write(os.path.join(self.__modelPath, "src", self.__RELMPolyFile) + "\n")
        
        # Close the file
        fhandle.close()


    #--------------------------------------------------------------------
    #
    # Prepare Matlab execution script and invoke the model.
    #
    # Input: None
    #        
    def run (self):
        """ Run STEP forecast."""


        # Check if Matlab format of the input catalog exists:
        if os.path.exists(self.inputCatalogFilename()) is False:
            
            # Convert catalog to Matlab format which is expected by the model
            CSEPGeneric.Catalog.toMatlab(OneDayForecast.inputCatalogFilename(self),
                                         self.inputCatalogFilename())

        
        dir_paths = [os.path.join(self.__modelPath, "HazFiles", "AllCal"),
                           os.path.join(self.__modelPath, "src")]

        fhandle = CSEPGeneric.openScript(dir_paths)

        # Set Matlab recursion limit 
        line = "set(0,'RecursionLimit', %s);\n" \
               %self.__args[STEPModel.__recursionLimitOption]
        fhandle.write(line)               
        
        # Invoke the model
        line = "AutoHazMain('%s');\n" %(self.parameterFile)
        fhandle.write(line)
        
        # Close the file
        CSEPGeneric.finalizeScript(fhandle)
        CSEPGeneric.invokeScript()
        
