"""
Module CSEPFactory
"""

__version__ = "$Revision: 3313 $"
__revision__ = "$Id: CSEPFactory.py 3313 2011-05-18 22:40:02Z  $"

import sys, re

from CSEPBorgIdiom import CSEPBorgIdiom
from RuntimeLoader import RuntimeLoader
import CSEPLogging, Environment, CSEP


#--------------------------------------------------------------------------------
#
# CSEPFactory.
#
# This class represents an interface for the object factory of registered Python 
# modules that are specific to the CSEP Natural Laboratory (NL).
# 
class CSEPFactory (CSEPBorgIdiom):

    # Static data of the class
    
    # Name of the element that defines geographical region for the testing center
    _geographicalRegion = 'geographicalRegion'
    
    _geographicalRegionDir = 'configDir'
    
    # Name of the element that defines namespace for all data products as
    # generated by the testing framework (used in filename formatting)
    _namespaceElement = 'namespace'
    
    # Element attributes of XML format configuration file
    __typeAttribute = 'type'

    __fileTypeAttribute = 'csepfile'
    __logger = None

      
    #----------------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #          config_file - Configuration file with installed Python modules
    #                         that represent the factory. 
    #          modules - Dictionary representing Python modules for the factory.
    #                    Default is an empty dictionary.       
    #          parent_class - A reference to the parent class for all classes 
    #                         within the factory. Default is None.
    #          
    # 
    def __init__ (self, 
                  config_file, 
                  modules = {},                  
                  parent_class = None):
        """ Initialization for CSEPFactory class"""

        CSEPBorgIdiom.__init__(self)
        
        if CSEPFactory.__logger is None:
           CSEPFactory.__logger = CSEPLogging.CSEPLogging.getLogger(CSEPFactory.__name__)
           
        # Build a dictionary of registered Python modules to generate objects.
        if '_pythonModules' in self.__dict__:
            self._pythonModules.update(modules)
        else:
            self._pythonModules = modules
        
        self.loadPythonModules(config_file,
                               parent_class)
     

    #----------------------------------------------------------------------------
    #
    # Return factory string identifier.
    # 
    # This method should be implemented by derived classes.
    #
    # Input: None
    # 
    # Output:
    #         Keyword identifying the factory.
    # 
    def type (self):
        """ Return type of the factory."""
              
        pass
     

    #---------------------------------------------------------------------------
    #
    # Create new object.
    #
    # Input:
    #          class_type - Keyword identifying Python module to use for object 
    #                       generation.
    #          input_variables - An optional list of input variables for new 
    #                            object. Default is None.
    # 
    # Output:
    #          Newly created object
    # 
    def object (self, class_type, input_variables = None):
        """ Instantiate object of specified type."""
        
        if self._pythonModules.has_key(class_type):
           if input_variables is None:
              return self._pythonModules[class_type]()
           
           else:
              if isinstance(input_variables, dict):
                  return self._pythonModules[class_type](**input_variables)
              elif isinstance(input_variables, list):
                  return self._pythonModules[class_type](*input_variables)
              else:
                  error_msg = "Unsupported type (%s) of input parameters %s is \
specified for %s keyword."  %(type(input_variables), input_variables, class_type)

           CSEPFactory.__logger.error(error_msg) 
           raise RuntimeError, error_msg
                  
           
        else:
           error_msg = "Unknown class identifier is specified by %s keyword. \
Supported classes are %s."  %(class_type, self.keys())

           CSEPFactory.__logger.error(error_msg) 
           raise RuntimeError, error_msg


    #----------------------------------------------------------------------------
    #
    # Return reference to the class
    #
    # Input:
    #          type - Keyword identifying Python class.
    # 
    # Output:
    #          Class reference that corresponds to the specified type.
    # 
    def classReference (self, type):
        """ Return reference to the class."""
        
        if type in self._pythonModules:
              return self._pythonModules[type]
           
        else:
           error_msg = "Unknown class identifier is specified by %s keyword. \
Supported classes are %s."  %(type, self.keys())

           CSEPFactory.__logger.error(error_msg) 
           raise RuntimeError, error_msg
        

    #----------------------------------------------------------------------------
    #
    # Get registered keys for the factory.
    #
    # Input: None.
    # 
    def keys (self):
        """ Returns existing keys for the CSEPFactory."""

        # Dictionary of registered Python modules to generate objects.        
        return self._pythonModules.keys()
        

    #----------------------------------------------------------------------------
    #
    # Add Python modules to the factory as specified by configuration file.
    #
    # Input:
    #          config_file - Configuration file with installed Python modules
    #                        to be added to the factory. 
    #          parent_class - A reference to the parent class for all classes 
    #                         within the factory. Default is None.
    # 
    def loadPythonModules (self, config_file, parent_class = None):
        """ Add Python modules to the factory as specified by configuration file."""

        
        import CSEPInitFile
        init_file = CSEPInitFile.CSEPInitFile(config_file) 
        
        if init_file.exists():

           from GeographicalRegions import Region

           # Set namespace for all data products generated by the testing 
           # framework
           namespace = init_file.elementValue(CSEPFactory._namespaceElement)
           if namespace is not None and len(namespace) != 0:
               CSEP.NAMESPACE = namespace
               
           # Set geographical region if it's specified
           __region_attribs = init_file.elementAttribs(CSEPFactory._geographicalRegion)
           __region_dir = __region_attribs.setdefault(CSEPFactory._geographicalRegionDir,
                                                      None)
           
           Region(__region_dir).set(init_file.elementValue(CSEPFactory._geographicalRegion))
 
           CSEPFactory.__logger.info("Loading modules from %s" \
                                     %config_file)   

           # Update Python search path before loading the modules
           loader = RuntimeLoader(init_file) 
           
           # Load modules specified by the file
           for module_elem, class_ref in loader.eachModuleClass(parent_class):   
              
              # Update module dictionary with class references if it's already
              # not there
              if class_ref.Type not in self._pythonModules:
                 # Type attribute must be defined for the class
                 self._pythonModules[class_ref.Type] = class_ref
                 

              # Check if any sub-elements (class data members) are provided:
              # each child represents static class attribute that would need 
              # to be changed for the run
              default_attrib = None # Value to use if attribute is missing
              for child in module_elem:
                 
                 # Strip namespace from the child's universal tag name:
                 # {Namespace}tag
                 tag = re.sub('{.*}', '', child.tag)
                 # Check if class contains specified data attribute
                 attr = getattr(class_ref, tag, default_attrib)
                 if attr is None:
                    
                    error_msg = "Class %s doesn't contain attribute: %s" \
                                %(class_ref, tag)
                              
                    CSEPFactory.__logger.error(error_msg)   
                    raise RuntimeError, error_msg

                 # Default type for the value is 'string'
                 value = child.text.strip()
                 
                 if CSEPFactory.__typeAttribute in child.attrib:
                    
                    # If element value is a filepath, check if CENTERCODE 
                    # is part of the path, replace with actual value if it's 
                    if child.attrib[CSEPFactory.__typeAttribute] == CSEPFactory.__fileTypeAttribute:
                        value = Environment.replaceVariableReference(Environment.CENTER_CODE_ENV,
                                                                     value)
                    else:
                        # Use eval to get actual value of specified type
                        value = eval("%s('%s')" %(child.attrib[CSEPFactory.__typeAttribute],
                                                value))
               
                 CSEPFactory.__logger.info("Setting %s attribute for %s class to %s" \
                                           %(tag, class_ref, value))
                    
                 setattr(class_ref, tag, value)
                    
        elif config_file is not None:
            # Raise an exception if specified file doesn't exist
            error_msg = "Specified configuration file %s does not exist." \
                        %config_file
                         
            CSEPFactory.__logger.error(error_msg)   
            raise RuntimeError, error_msg
        