"""
Module CSEPLogging
"""

__version__ = "$Revision: 3312 $"
__revision__ = "$Id: CSEPLogging.py 3312 2011-05-17 22:55:22Z  $"

import sys, logging

from CruiseControlOptionParser import CruiseControlOptionParser


#--------------------------------------------------------------------------------
#
# CSEPLogging
#
# This class represents an interface for the logging functionality of CSEP.
# It is implemented as a singleton to guarantee proper configuration for the
# logging module upon first call to access any logger from it.
# 
class CSEPLogging (object):

    # Static data of the class 

    # Singleton instance
    __isInitialized = False
    
      
    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input:
    #        filename - Filename to redirect logging messages to.
    #        log_level - Logging level.   
    # 
    def __init__ (self, filename, log_level):
        """ Initialization for CSEPLogging class"""

        # Since CSEPFactory derived classes initialize their logger before any
        # command-line arguments are parsed, need to check if logging redirect to
        # the file is specified - subsequent calls to logging.basicConfig
        # have no effect if StreamHandler is already generated by default 
        # on first call to the logging
        parser = CruiseControlOptionParser()
              
        # Set up the root logger
        date_format = '%a, %Y-%m-%d %H:%M:%S'
        message_format = '%(asctime)s %(name)s %(levelname)-8s %(message)s'
       
        if filename is not None:
           # Redirect all progress messages to the file:
           # Subsequent calls to basicConfig have no effect if StreamHandler is
           # initialized on the very first call to the module
           logging.basicConfig(format = message_format,
                               level = log_level,
                               datefmt = date_format,
                               filename = filename,
                               filemode = 'w')
        else:
           # Redirect all progress messages to the stdout
           logging.basicConfig(format = message_format,
                               level = log_level,
                               datefmt = date_format,
                               stream = sys.stdout)
        

    #--------------------------------------------------------------------
    #
    # Get instance of the class.
    #
    # Input:
    #        name - Name of the logger. Default is the root logger name -
    #               empty string.
    #        log_level - Logging level. Default is logging.INFO.
    #        filename - Filename to redirect logging messages to. Default is
    #                   None. 
    #
    @staticmethod
    def getLogger (name = '', log_level = logging.INFO, filename = None):
        """ Get specified instance of the logger. This method configures
            Python's logging module if it was not configured yet, and returns
            a reference to specified logger instance."""
        
        if (CSEPLogging.__isInitialized is False) or (filename is not None):
           
           CSEPLogging.frame(CSEPLogging) 
           CSEPLogging(filename, log_level)
           CSEPLogging.__isInitialized = True
        
        return logging.getLogger(name)


    #===========================================================================
    # Helper function to get information of current frame in calling stack
    #
    # Input: 
    #        cls - Class from which method is called
    #        level - Frame level. Default is 1.
    #
    #===========================================================================
    @staticmethod
    def frame(cls, level=1):
        """ Get frame information at 'level' (default is 1) in calling stack.
            Returned string contains filename, line number and class method name."""
        
        obj = sys._getframe(level)
        
        return'File %s line=%s %s.%s()' %(obj.f_code.co_filename,
                                          obj.f_lineno,
                                          cls.__name__,
                                          obj.f_code.co_name)


# Invoke the module
if __name__ == '__main__':

   
    import sys, logging
   
    # Set up the root logger
    date_format = '%a, %Y-%m-%d %H:%M:%S'
    message_format = '%(asctime)s %(name)s %(levelname)-8s %(message)s'

    logging.basicConfig(format = message_format,
                        level = logging.INFO,
                        datefmt = date_format,
                        stream = sys.stdout)

   
   # Invoke all tests
    CSEPLogging.getLogger().info("some test info")
        
# end of main
