"""
Module CSEPSchedule
"""

__version__ = "$Revision: 5012 $"
__revision__ = "$Id: CSEPSchedule.py 5012 2015-01-15 19:11:52Z liukis $"

import datetime
import CSEPXML, CSEPLogging


#--------------------------------------------------------------------------------
#
# CSEPSchedule
#
# This module represents a schedule for the forecast groups:
# to invoke forecast models
# to invoke evaluation tests for existing forecast models.
#
class CSEPSchedule (object):

    # Static data members

    __logger = None

    # Value separator for range format:
    # 1985:1990   - values from 1985 up to 1990 inclusive with default increment of 1
    # 1985:1990:2 - values from 1985 up to 1990 inclusive with specified increment of 2
    __rangeSeparator = ':'
    
    # Default value for range increment if not provided by configuration file
    __defaultRangeIncrement = 1
    
    # Class to represent days of the month and time within those days
    # (introduced to support 30-minutes forecast class within CSEP)
    class DayAndTime (list):
        
        def __init__ (self,
                      days=[],
                      hour_txt=None,
                      minute_txt=None,
                      second_txt=None):
            
            # Construct list
            list.__init__(self, days)
            
            # Default: time is not specified
            self.timeDict = {}
            
            self.extendTime(days,
                            hour_txt,
                            minute_txt,
                            second_txt)
        
            
        def __repr__ (self):
            """ String representation of the object"""
            return "%s:<<%s>>" %(list(self), self.timeDict)
        
        
        def extend (self,
                    days, 
                    hours=None, 
                    minutes=None, 
                    seconds=None):
            """ Extend list of days with additional days and corresponding times if any"""
            
            for each_day in days:
                if each_day not in self:
                    self.append(each_day)
            
            self.extendTime(days,
                            hours, 
                            minutes, 
                            seconds)
            
            # If days is an "DayAndTime" object, 
            # then update internal dictionary of times
            if isinstance(days, CSEPSchedule.DayAndTime):
                # self.timeDict.update(days.timeDict)
                for each_day, each_time in days.timeDict.iteritems():
                    # iterate over days
                    self_time = self.timeDict.setdefault(each_day, {})
                    
                    for each_hour, each_min in each_time.iteritems():
                        self_hour = self_time.setdefault(each_hour, {})
                        
                        for each_min, each_secs in each_min.iteritems():
                            self_hour.setdefault(each_min, []).extend(each_secs)
                            
            
        def extendTime (self,
                        days,
                        hour_txt=None,
                        minute_txt=None,
                        second_txt=None):
            """ Extend list of times with additional times if any"""
            
            for each_day in days:

                # Capture time information for the days if provided
                for hour in CSEPSchedule.rangeValues(hour_txt):
    
                    for minute in CSEPSchedule.rangeValues(minute_txt):
                   
                       for second in CSEPSchedule.rangeValues(second_txt):
                           self.timeDict.setdefault(each_day, {}).setdefault(hour, {}).setdefault(minute, []).extend(CSEPSchedule.rangeValues(second))
            
    
    #--------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #       any_value - Character that represents 'any' value. Default is
    #                   CSEPXML.ANY_VALUE. 
    # 
    def __init__ (self, any_value = CSEPXML.ANY_VALUE):    
        """ Initialization for CSEPSchedule class."""
    
        if CSEPSchedule.__logger is None:
           CSEPSchedule.__logger = CSEPLogging.CSEPLogging.getLogger(CSEPSchedule.__name__)
           
        # Remember 'any' value
        self.__anyValue = any_value 
        
        # Flag to indicate that dictionary was set to the default state
        self.__initialized = True
        
        # Schedule dictionary of dictionaries in the format:
        # year : {month : [days]} 
        self.__dict = {any_value: {any_value: CSEPSchedule.DayAndTime([any_value])}}


    #--------------------------------------------------------------------
    #
    # Add element to the dictionary.
    # 
    # Input: 
    #        year - Years.
    #        month - Months within the year.
    #        days - Days for specified months.
    #        hour - Hour for specified days.
    #        minute - Minute for specified hour.
    #        second - Second for specified minute.
    #
    # Output: None.
    #
    def add (self, 
             years, 
             months, 
             days,
             hour=None,
             minute=None,
             second=None):
        """ Add an entry to the schedule."""


        if self.isDefault() is True:
           # This is the first time entry is added to the dictionary,
           # remove default setting
           self.__dict = {}
           self.__initialized = False
        
        
        year_list = years.split()
        month_list = months.split()

        for each_year in year_list:
           CSEPSchedule.__logger.debug("Adding year=%s month=%s days=%s (hr=%s min=%s sec=%s)" \
                                       %(each_year, months, days,
                                         hour, minute, second))
           
           # Support range of years if any is provided
           for year in CSEPSchedule.rangeValues(each_year):

               # Support range of months if any is provided
               for each_month in month_list:
                   
                   for month in CSEPSchedule.rangeValues(each_month):
                       
                       # Append specified days and time to the month:
                       for each_day in days.split():
                           day_list = CSEPSchedule.rangeValues(each_day)
                           self.__dict.setdefault(year, {}).setdefault(month, 
                                                                       CSEPSchedule.DayAndTime()).extend(CSEPSchedule.DayAndTime(day_list,
                                                                                                                                 hour,
                                                                                                                                 minute,
                                                                                                                                 second))
        # print "DICT: ", self.__dict
        

    #--------------------------------------------------------------------
    #
    # Return values that correspond to specified range, or return original
    # value as a list if it does not represent range of values.
    # 
    # Input: 
    #        value - Value as provided in configuration file
    #
    # Output: None.
    #
    @staticmethod
    def rangeValues(value):
        """ Return values that correspond to specified range, or return original
            value as Python's list if it does not represent range of values."""
        
        # Return an empty list if None value is provided    
        if value is None:
            return []
        
        value = value.strip()
        return_values = [value]
        
        # Range is provided, iterate over specified values
        if CSEPSchedule.__rangeSeparator in value:
            value_str = value.split(CSEPSchedule.__rangeSeparator)
               
            increment = CSEPSchedule.__defaultRangeIncrement
            if len(value_str) == 3:
                increment = int(value_str[-1])
        
            return_values = range(int(value_str[0]),
                                  int(value_str[1])+1,
                                  increment)
        
        return [str(each_value) for each_value in return_values]
        

    #--------------------------------------------------------------------
    #
    # Checks if specified date is within the schedule.
    # 
    # Input: 
    #        test_date - datetime object that represents test date.
    #
    # Output: True if date is within the schedule, False otherwise.
    #
    def has (self, test_date):
        """ Checks if specified date is within the schedule."""
     
        # If schedule was set as a default, then any date is within the schedule.
        if self.isDefault() is True:
           return True

        #print self.__dict
        # print "TestDate=", test_date, "SCHED:", self.__dict
        
        # Dictionary of matching years
        found_years = {}
        
        # Convert year to string: dictionary stores info in string format
        year_string = str(test_date.year)
        
        if year_string in self.__dict:
           found_years[year_string] = self.__dict[year_string]
           
        if self.__anyValue in self.__dict:
           found_years[self.__anyValue] = self.__dict[self.__anyValue]
        
        # Did not find matching or "any" year 
        if len(found_years) == 0:
           return False
        
        # print "FoundYears:", found_years
        
        # Dictionary of matching months        
        found_months = {}

        # Convert month to string: dictionary stores info in string format
        month_string = str(test_date.month)

        # Search month dictionaries (that might represent specific year or "any" year)
        for month_dict in found_years.values():
           
           if month_string in month_dict:
              found_months.setdefault(month_string, CSEPSchedule.DayAndTime()).extend(month_dict[month_string])
              
           if self.__anyValue in month_dict:
              found_months.setdefault(self.__anyValue, CSEPSchedule.DayAndTime()).extend(month_dict[self.__anyValue])

        #print "FoundMonths:", found_months
        
        # Did not find matching or "any" month 
        if len(found_months) == 0:
           return False
        
        # String representation of test day and time
        day_str = str(test_date.day)
        hour_str = str(test_date.hour)
        minute_str = str(test_date.minute)
        second_str = str(test_date.second)
        #print "STR of time:", hour_str, minute_str, second_str
        
        
        for day_list in found_months.values():
           
           #print "DAY_LIST:", day_list

           found_day = None
           
           # Has any day set
           if self.__anyValue in day_list:
               found_day = self.__anyValue

           else:
               # Has requested day set (check for string representation of day)
               found_day = [s for s in day_list if s == day_str]
               if len(found_day):
                   found_day = found_day[0]
                   
               else:
                   found_day = None 
           
        
           # Day matched
           if not (found_day is None):
                
                #print "FOUND:", found_day, "in dict:", found_day in day_list.timeDict
                #print "DICT:", day_list.timeDict, "months:", found_months
                
                # and time is not set
                if not (found_day in day_list.timeDict) or \
                   len(day_list.timeDict[found_day]) == 0:
                   
                   # print "Time:", test_date.hour, test_date.minute, test_date.second 
                   if test_date.hour != 0 or \
                      test_date.minute != 0 or \
                      test_date.second != 0:
                       
                       return False
                   
                   else:
                       # Start time of '00:00:00" is assumed when time is not specified
                       return True
                
                # time is set for the day
                else:
                     time_dict = day_list.timeDict[found_day]
                     if hour_str in time_dict and \
                        minute_str in time_dict[hour_str] and \
                        second_str in time_dict[hour_str][minute_str]:
                         return True
                     
                     else:
                         return False 
           
        # Have not found day for specific year and month
        return False


    #---------------------------------------------------------------------------
    #
    # Flag if schedule is set to default: any date of any year.
    # 
    # Input: None
    #
    # Output: True if schedule is set to the default one (any date of any year).
    #
    def isDefault (self):
        """ Flag to specify that schedule was set to default one (any date)."""
      
        return self.__initialized is True 


    #----------------------------------------------------------------------------
    #
    # Generator method that iterates through all dates of the schedule.
    # 
    # Input: None
    #
    # Output: Next date in the schedule
    #
    def dates (self, 
               start_date = None, 
               stop_date = None):
        """ Generator to return each date from the schedule in chronological order."""
      
        import calendar
        
        schedule_calendar = calendar.Calendar()
        
        all_years = None
        
        # ATTN: the only condition is the start date must be explicitly set
        #       for the schedule - to know where to start iteration from
        if self.isDefault() is True or \
           self.__anyValue in self.__dict:
      
           # Check if start date is provided
           if start_date is not None:
               
               # Issue a warning that start_date.year is used as starting point,
               # and 10 years since that start date is assumed to be the end year
               # for the schedule
               stop_year = start_date.year + 10
               if stop_date is not None:
                   stop_year = stop_date.year + 1
                   
               all_years = range(start_date.year, stop_year)
               
               msg = "%s: Year of provided start date (%s) is used since schedule provides ambiguous \
start date ('%s' value is specified for the year) within generator method" %(CSEPLogging.CSEPLogging.frame(CSEPSchedule),
                                                                             start_date,
                                                                             self.__anyValue)
        
               CSEPSchedule.__logger.warning(msg)
               
           else: 
               error_msg = "dates(): Start date is ambiguous for schedule \
(has '%s' value for the year) within generator method. Please provide start date." %self.__anyValue
    
               CSEPSchedule.__logger.error(error_msg)
               raise RuntimeError, error_msg
           
        else:
            # Sort years of the schedule
            all_years = [int(year) for year in self.__dict.keys()]
            all_years.sort()

        # print "YEARS:", all_years
        for each_year in all_years:

           # Find all months that were specified for the year 
           year_key = self.__anyValue
           
           if self.__anyValue not in self.__dict:
               # Dictionary of months for the year is provided
               year_key = str(each_year)
          
           all_months = self.__dict[year_key]
           
           # Sort months and days (default is "all" months are specified)
           months = range(1, 13)
           
           # "Any" month is specified
           if self.__anyValue not in all_months:
              months = [int(month) for month in all_months.keys()]
              months.sort()
        
           #print "MONTHS:", all_months      
           # Step through "day" list for each month: "months" list is set to 
           # integer values for identified months within a year
           for each_month in months:

              # Identify which month key to use to access schedule information 
              month_key = self.__anyValue
              if self.__anyValue not in all_months:
                  month_key = str(each_month)
                
              all_days = all_months[month_key]
              time_info = all_days.timeDict
              
              #print "TIME_INFO:", time_info
              
              # Any day is specified for the month
              if self.__anyValue in all_days:
                 
                 # Find all valid days within a month: "0" is returned for day
                 # outside of the month but that is part of month week
                 all_days = [d for d in schedule_calendar.itermonthdays(each_year, 
                                                                        each_month) if d != 0]
              else:
                 all_days = [int(d) for d in all_days] 
              
              all_days.sort()
              
              #print "DAYS:", all_days
              for each_day in all_days:

                 # Return current date within the schedule
                 next_date = datetime.datetime(each_year,
                                               each_month,
                                               each_day)
              
                 # Time is provided 
                 day_key = str(each_day)
                 # Any day is specified for the month
                 if self.__anyValue in all_months[month_key]:
                     day_key = str(self.__anyValue)
                     
                 if day_key in time_info and len(time_info[day_key]):
                     all_hours = range(0, 24)
                     
                     if self.__anyValue not in time_info[day_key]:
                         # Explicit hours are provided
                         all_hours = [int(hour) for hour in time_info[day_key].keys()]
                         all_hours.sort()
                    
                     #print "DAY=:", each_day, "hours: ", all_hours
                     for each_hour in all_hours:
                         
                         hour_key = self.__anyValue
                         if self.__anyValue not in time_info[day_key]:
                             hour_key = str(each_hour)
                     
                         all_minutes = range(0, 60)
                         if self.__anyValue not in time_info[day_key][hour_key]:
                             all_minutes = [int(minute) for minute in time_info[day_key][hour_key].keys()]
                             all_minutes.sort()
                             
                         for each_minute in all_minutes:
                             minute_key = self.__anyValue
                             if self.__anyValue not in time_info[day_key][hour_key]:
                                 minute_key = str(each_minute)
                             
                             # Access seconds
                             for each_second in time_info[day_key][hour_key][minute_key]:
                                 next_date = datetime.datetime(each_year,
                                                               each_month,
                                                               each_day,
                                                               each_hour,
                                                               each_minute,
                                                               int(each_second)) 

                                 if (start_date is not None) and (next_date < start_date):
                                     continue
                                  
                                 if (stop_date is not None) and (next_date > stop_date):
                                     # Stop generator
                                     return
                                 
                                 else:
                                     yield next_date

                 else:   

                     if (start_date is not None) and (next_date < start_date):
                         continue
                      
                     if (stop_date is not None) and (next_date > stop_date):
                         # Stop generator
                         return
                     
                     else:
                         yield next_date
              
        # Stop generator
        return


# Can't invoke the module - expects Python objects as inputs

