"""
Module ForecastFacade
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import os

import CSEP, CSEPLogging, CSEPFile, CSEPGeneric
from Forecast import Forecast
from CSEPStorage import CSEPStorage
from ForecastHandlerFactory import ForecastHandlerFactory


#-------------------------------------------------------------------------------
#
# ForecastFacade.
#
# This class represents "Facade" interface for existing forecast files within
# testing framework. It is introduced to support multiple forecasts formats.
# 
class ForecastFacade (CSEPStorage):
    
    # Static data
    __logger = None
    
    
    #---------------------------------------------------------------------------
    # Initialize
    #---------------------------------------------------------------------------
    def __init__ (self, forecast_dir):
        """ Initialize object of the class."""

        # Initialize base class with directory for the forecasts files
        CSEPStorage.__init__(self,
                             forecast_dir)
        
        self.__dir = forecast_dir
        self.__files = []
        self.__ignoreExtension = None
        self.__ignoreName = None
        
        if ForecastFacade.__logger is None:
            ForecastFacade.__logger = CSEPLogging.CSEPLogging.getLogger(ForecastFacade.__name__)


    #---------------------------------------------------------------------------
    #
    # Checks if file of interest exists: in 'forecasts' or in 
    # 'forecasts/archive/YYYY_MM' directory. 
    #
    def fileExists (self,
                    forecast_file,
                    archive_dir):
       """ Checks if file of interest exists: in 'forecasts' or in 
           'forecasts/archive/YYYY_MM' directory.
           Inputs:
           forecast_file - Forecast file of interest
           archive_dir - Archive directory for the file of interest.
           
           Returns: True if file exists, False otherwise.
           """
       
       return (os.path.exists(forecast_file) is True or \
               (archive_dir is not None and \
                CSEPStorage.stage(self, 
                                  [os.path.basename(forecast_file)],
                                  archive_dir) is True))


    #---------------------------------------------------------------------------
    # Returns list of existing forecasts files.
    #---------------------------------------------------------------------------
    def allFiles (self,
                  archive_dir,
                  post_process,
                  ignore_name = None,
                  ignore_extension = None):
        """ Returns list of existing forecasts files.
            Inputs:
            archive_dir - archive directory for existing forecasts
            post_process - PostProcess object for the forecast group
            ignore_name - Filename to ignore from listing
            ignore_extension - List of file extensions to ignore from listing
        """

        # Identify available forecasts
        self.__files = []
        self.__ignoreExtension = ignore_extension
        self.__ignoreName = ignore_name
        
        dir_list = os.listdir(self.__dir)
        
        # The same directory listing returns files in different order on 
        # different machines - makes acceptance tests to fail. 
        dir_list.sort()

        # Archive directory to check for already existing data files if any
        model_archive_dir = None

        for model in dir_list:        
            model_path = os.path.join(self.__dir, 
                                      model)

            if Forecast.isFile(model_path, 
                               ignore_name, 
                               ignore_extension):
               
                # Construct archive directory for the model
                if model_archive_dir is None and \
                   Forecast.archiveDir(model_path) is not None:
                   
                   model_archive_dir = os.path.join(archive_dir,
                                                    Forecast.archiveDir(model_path))
                
                register_file = self.basedOnXMLTemplate(model_path,
                                                        model_archive_dir,
                                                        post_process)
                      
                # File was not processed
                if not register_file:
                    # ForecastFacade.__logger.info('Entering ASCII: %s' %register_file)
                    
                    #===========================================================
                    ### XML forecast template is disabled
                    #===========================================================
                    ascii_path = CSEPFile.Name.ascii(model_path)
                    register_file = CSEP.Forecast.compressedFilename(ascii_path)

                    ForecastFacade.__logger.info("Final format of forecast file '%s' for model %s"
                                                 %(register_file, 
                                                   model))
                    if not self.fileExists(register_file,
                                           model_archive_dir):
                    
                        # Check if ASCII format file of the forecast exists
                        ForecastFacade.__logger.info("ASCII format of forecast file '%s' for model %s"
                                                    %(ascii_path, 
                                                      model))
                        
                        if not self.fileExists(ascii_path,
                                               model_archive_dir):
        
                           # Convert XML format of the forecast to ASCII
                           if CSEPFile.Extension.toFormat(model_path) == CSEPFile.Format.XML:
                               _model = ForecastHandlerFactory().CurrentHandler.XML(model_path)
                               _model.toASCII(ascii_path)
        
                           elif CSEPFile.Extension.toFormat(model_path) == CSEPFile.Format.MATLAB:
                               # Convert Matlab format forecast to ASCII
                               CSEPGeneric.Forecast.toASCII(model_path,
                                                            result_file = ascii_path)
                               
                           else:
                               error_msg = 'Unknown forecast format %s is provided, one of %s is supported' \
                                           %(model_path, (CSEPFile.Format.ASCII,
                                                          CSEPFile.Format.XML, 
                                                          CSEPFile.Format.MATLAB))
                               ForecastFacade.__logger.error(error_msg)
                               raise RuntimeError, error_msg 
        
                           # Create metadata file for the forecast 
                           comment = "Forecast file in %s format" %(CSEPFile.Format.ASCII) 
                          
                           Forecast.metadata(ascii_path,
                                             comment,
                                             model_archive_dir)
                        
                        # Check if compression is enabled
                        if CSEP.Forecast.UseCompression:
                            # Compress final forecast file
                            archive = CSEPFile.GZIPArchive(register_file,
                                                           CSEPFile.Mode.WRITE)
                            archive.add(ascii_path)
                            del archive # Force file closure
                            
                            # Create metadata file for the forecast 
                            comment = "Archive of %s forecast file in %s format" \
                                       %(ascii_path,
                                         CSEPFile.Extension.toFormat(register_file))
                            
                            Forecast.metadata(register_file,
                                              comment,
                                              model_archive_dir)
                                                   
                            # Remove uncompressed file
                            ForecastFacade.__logger.info("Removing data file %s after archiving it to %s" %(ascii_path,
                                                                                                            register_file))
                            os.remove(ascii_path)

                
                # Register the file and continue   
                register_file = os.path.basename(register_file)
                not_registered = (self.__files.count(register_file) == 0)
               
                if not_registered:
                   self.__files.append(register_file)
                  
        return self.__files      


    #---------------------------------------------------------------------------
    # Process existing forecast file if master XML template is enabled for the
    # experiment
    #---------------------------------------------------------------------------
    def basedOnXMLTemplate (self,
                            model_path,
                            model_archive_dir,
                            post_process):
        
       # Master XML template is disabled return None for filename        
       if not CSEP.Forecast.UseXMLMasterTemplate:
           return None
       
        
       # Check if corresponding file based on XML forecast
       # template exists (file was created by 
       # populating master XML template)
       # File is an archive if compression is enabled
       #
       from_xml_path = CSEP.Forecast.compressedFilename(CSEP.Forecast.fromXMLTemplateFilename(model_path))
       
       ForecastFacade.__logger.info("Checking for %s file for '%s' model." %(from_xml_path,
                                                                             model_path))

    
       #============================================================
       # Final format of the model based on XML template does not 
       #============================================================
       if not self.fileExists(from_xml_path,
                              model_archive_dir):
          
          ForecastFacade.__logger.info("Creating %s file for '%s' \
based on XML template." %(from_xml_path,
                          model_path))
          
          xml_path = CSEPFile.Name.xml(model_path)
          ForecastFacade.__logger.info("Forecast XML file '%s' for model %s" \
                                       %(xml_path, 
                                         model_path))
          
          # Fix for Trac ticket #177
          # If XML format of the forecast exists, make sure that it's
          # the one generated by the CSEP testing framework (to guarantee
          # that forecast passes through master ForecastML template)
          if self.fileExists(xml_path,
                             model_archive_dir):
              
               _val_file = ForecastHandlerFactory().CurrentHandler.XML(xml_path)
               _val_file.validate(post_process,
                                  model_archive_dir)
          
          else:
             #======================================================
             # XML format of the model does not exist, create one
             #======================================================
             # Check for ASCII format forecast file existence - 
             # it will be required to create XML format forecast
             ascii_path = CSEPFile.Name.ascii(model_path)
             
             if not self.fileExists(ascii_path,
                                    model_archive_dir):
                
                # Forecast is in Matlab format, convert to ASCII 
                CSEPGeneric.Forecast.toASCII(model_path)
                
                # Create metadata file for the forecast 
                comment = "Forecast file in %s format used to create XML format forecast file" \
                           %CSEPFile.Format.ASCII
                
                Forecast.metadata(ascii_path,
                                  comment,
                                  model_archive_dir)
                                      
             # Create XML format file of the forecast
             template = ForecastHandlerFactory().CurrentHandler.XML(post_process.template)
             xml_path = template.toXML(ascii_path, 
                                       post_process.start_date,
                                       post_process.end_date,
                                       os.path.basename(model_path))
             
             # Create metadata file for the forecast 
             comment = "Forecast file in %s format that is based on XML master template %s" \
                        %(CSEPFile.Format.XML, 
                          post_process.template)
             
             Forecast.metadata(xml_path,
                               comment,
                               model_archive_dir)
             
          
          # Convert XML format forecast to ASCII or HDF5
          _model = ForecastHandlerFactory().CurrentHandler.XML(xml_path)
          final_file = _model.toASCII(CSEP.Forecast.fromXMLTemplateFilename(model_path))
         
          if CSEP.Forecast.UseCompression:
              # Compress final forecast file
              archive = CSEPFile.GZIPArchive(from_xml_path,
                                             CSEPFile.Mode.WRITE)
              archive.add(final_file)
              del archive
              
              # Create metadata file for the forecast 
              comment = "Archive of %s forecast file in %s format" \
                        %(final_file,
                          CSEPFile.Extension.toFormat(from_xml_path))
                
              Forecast.metadata(from_xml_path,
                                comment,
                                model_archive_dir)
              
              # Remove uncompressed file
              ForecastFacade.__logger.info("Removing data file %s after archiving it to %s." %(final_file,
                                                                                               from_xml_path))
              os.remove(final_file)
              final_file = from_xml_path
            
          # Create metadata file for the forecast 
          comment = "Forecast file in %s format that is based on XML master template %s" \
                     %(CSEPFile.Extension.toFormat(final_file), 
                       post_process.template)
          
          Forecast.metadata(final_file,
                            comment,
                            model_archive_dir)

          # ForecastFacade.__logger.info('Returning FINAL FILE from XML: %s' %final_file)
          return final_file
      
       # File already exists, return it's name
       return from_xml_path
                
