"""
Module CSEPTestCase
"""

__version__ = "$Revision: 4149 $"
__revision__ = "$Id: CSEPTestCase.py 4149 2012-12-19 03:06:30Z liukis $"


import sys, string, os, datetime, unittest, copy

import CSEP, EvaluationTest
from cseprandom import CSEPRandom
from Environment import *
from CSEPLogging import CSEPLogging
from DataSourceFactory import DataSourceFactory
from EvaluationTest import EvaluationTest
from RateForecastHandler import RateForecastHandler
from ForecastHandlerFactory import ForecastHandlerFactory


#--------------------------------------------------------------------------------
#
# CSEPTestCase.
#
# This class is designed as a fixture for any CSEP test case. 
#
class CSEPTestCase (unittest.TestCase):

    # Static data
    
    # Optional environment variable to specify location of runtime results
    # directories generated by CSEP acceptance and unit tests
    ResultsDirVar = 'CSEP_RESULTS_DIR'
    
    # Test date
    Date = datetime.datetime(2006, 11, 2)

    # Test date
    CumulativeStartDate = datetime.datetime(2006, 1, 1)
    
    # Top-level directory for the tests
    __Dir = os.path.join(Environment.Variable[CENTER_CODE_ENV], 
                         'src', 'generic', 'test')
    
    # Directory that stores reference data. It is being used by the comparison 
    # tools to validate test results.
    ReferenceDataDir = os.path.join(__Dir, 'data')

    # Directory for the single test
    TestDirPath = os.path.join(__Dir, 'results')

    # Name of the test currently being invoked
    __TestName = ''    
    
    # ANSS data source with default start date using options to:
    # * disable download of raw data
    # * disable pre-processing of raw data
    DataSourceArgs = {'download_data' : False,
                      'pre_process_data' : False}
   
    # Flag if static data (such as ForecastFactory) has been initialized by
    # acceptance tests configuration
    __initialized = False
    
    # Configuration file for ForecastFactory used by acceptance tests
    ForecastFactoryConfigFile = "TestForecastFactory.init.xml"
    
   
    #--------------------------------------------------------------------
    #
    # Set up testing scenario.
    #
    # Input: 
    #        None.
    # 
    def setUp (self):
        """ Set up routine for the CSEPTestCase class."""
        
        CSEPTestCase.setDataSource()
        
        if os.environ.has_key(CSEPTestCase.ResultsDirVar) is True:
            
            CSEPTestCase.TestDirPath = os.path.join(os.environ[CSEPTestCase.ResultsDirVar],
                                                    'results')
            
        CSEPLogging.getLogger(CSEPTestCase.__name__).debug("Using %s directory for the test results" 
                                                           %CSEPTestCase.TestDirPath)
                
        if (os.path.exists(CSEPTestCase.TestDirPath) is False):
            CSEPLogging.getLogger(CSEPTestCase.__name__).debug("Creating test directory '%s'..." \
                                                               %CSEPTestCase.TestDirPath)

            os.makedirs(CSEPTestCase.TestDirPath)

        ForecastHandlerFactory().object(RateForecastHandler.Type)
        
        # Initialize CSEP environment
        self.initialize()
        
        # Initialize factories for tests
        CSEPTestCase.__initializeFactories()
        

        
    #--------------------------------------------------------------------
    #
    # Clean up test case scenario by storing testing scenario results.
    #
    # Input: 
    #        None.
    # 
    def tearDown (self):
        """ Cleanup routine for the CSEPTestCase class."""

        # Check existence of the test directory
        if os.path.exists(self.TestDirPath) is True:
                       
           # move test directory to the unique location to
           # preserve result files for manual examination
           new_dir = "%s-%s" %(self.TestDirPath, self.__TestName)
           new_location = os.path.join(self.__Dir, new_dir)
           
           CSEPLogging.getLogger(CSEPTestCase.__name__).debug("Saving test directory %s to \
new location %s..." %(self.TestDirPath, new_location))

           os.rename(self.TestDirPath, new_location)


    #--------------------------------------------------------------------
    #
    # Set up name for the test scenario.
    #
    # Input: 
    #        name - Test name for the scenario.
    # 
    def setTestName (self, name):
        """ Set up name for the test scenario."""
        
        self.__TestName = name
        
        # Report the test scenario to the user
        CSEPLogging.getLogger(CSEPTestCase.__name__).debug("Invoking %s..." \
                                                           %(self.__TestName))


    #--------------------------------------------------------------------
    #
    # Set up authorized data source for the test scenario.
    #
    # Input: 
    #        source - Data source keyword. Default is DataSourceFactory.DefaultType
    # 
    @staticmethod
    def setDataSource (source=DataSourceFactory.DefaultType,
                       args = None):
        """ Set up authorized data source for the test scenario."""

        # Report the test scenario to the user
        CSEPLogging.getLogger(CSEPTestCase.__name__).debug("Setting %s data source" \
                                                           %(source))
        DataSourceFactory.composite().clear()
        
        vars = copy.deepcopy(CSEPTestCase.DataSourceArgs)
        if args is not None:
            vars.update(args)
            
        DataSourceFactory().object(source,
                                   input_variables = vars)
        

    #--------------------------------------------------------------------
    #
    # Initialization routine for Matlab.
    #
    # Input: None
    # 
    # Output: None
    #
    def initialize (self):
        """ Initialize CSEP environment for the test."""

        
        CSEP.DebugMode = True
        
        EvaluationTest.initialize(num_test_simulations = 10)
        CSEPRandom.ReadSeedFromFile = True
        
        # Forecast related settings: 
        # forecast_weights = True
        # forecast_xml_template = False
        # validate_xml_forecast = False
        # forecast_map = False
        CSEP.Forecast.initialize(validate_xml_forecast = False)
        
        # Catalog related settings
        CSEP.Catalog.initialize(num_decluster_simulations = 10,
                                num_catalog_variations = 10)


    #--------------------------------------------------------------------
    #
    # Initialization routine for Matlab.
    #
    # Input: None
    # 
    # Output: None
    #
    @staticmethod
    def __initializeFactories ():
        """ Initialize CSEP factories for the test."""
        
        if CSEPTestCase.__initialized is False:
           from ForecastFactory import ForecastFactory

           ForecastFactory(os.path.join(CSEPTestCase.ReferenceDataDir, 
                                        CSEPTestCase.ForecastFactoryConfigFile))

           CSEPTestCase.__initialized = True
     

    #----------------------------------------------------------------------------
    #
    # Utility to calculate percent difference for two values.
    #
    @staticmethod
    def percentDiff(value1, value2):
        """ Utility to calculate percent difference for two values."""

        diff = value1 - value2
                 
        if (value1 + value2) != 0.0:
           # percent_diff = 100*diff/((value1+value2)/2.0)
           diff *= 200.0/(value1 + value2)

        return abs(diff) 
     
     