"""
Module CatalogPostProcessingTest
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import sys, os, unittest, shutil, datetime

import CSEPFile, CSEP, GeoNetNZDataSource, \
       GeographicalRegions, Environment

from CSEPTestCase import CSEPTestCase
from RELMCatalog import RELMCatalog
from CatalogDataSource import CatalogDataSource
from DataSourceFactory import DataSourceFactory
from OneDayModelPostProcess import OneDayModelPostProcess
from OneDayModelInputPostProcess import OneDayModelInputPostProcess
from PostProcessFactory import PostProcessFactory
from CSEPLogging import CSEPLogging


 #-------------------------------------------------------------------------------
 #
 # Test post-processing of GeoNetNZ catalog data used for the forecast generation 
 # and evaluation tests. These tests validate generated catalog uncertainties as well.
 #
class CatalogPostProcessingTest (CSEPTestCase):

    # Reference data directory
    __referenceDataDir = os.path.join(Environment.Environment.Variable[Environment.CENTER_CODE_ENV],
                                      'src',
                                      'SCECModels',
                                      'NewZealand',
                                      'test',
                                      'data',
                                      'postProcess')

    
    #----------------------------------------------------------------------------
    #
    # Test post-processing of GeoNetNZ catalog data used for one-day forecast 
    # evaluation tests, and evaluate generated catalog uncertainties.
    #
    def testOneDayModelPostProcessing(self):
        """ Test post-processing of GeoNet NZ catalog for one-day \
forecast evaluation as well as catalog uncertainties, and succeed. """

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 "GeoNetNZ-OneDayModelPostProcessing")

        one_day_model_post_process_minMagnitude = OneDayModelPostProcess.MinMagnitude
        one_day_model_post_process_maxDepth = OneDayModelPostProcess.MaxDepth

        try:  
    
            OneDayModelPostProcess.MinMagnitude = 3.0
            OneDayModelPostProcess.MaxDepth = 40.0
            uncertainties_reference_dir = "OneDayModel_uncertainties"
              
            # Invoke post-processing test for one day forecast evaluation
            self.__postProcessing(OneDayModelPostProcess.Type,
                                  datetime.datetime(2008, 9, 7),
                                  uncertainties_reference_dir)

        finally:
            
            OneDayModelPostProcess.MinMagnitude = one_day_model_post_process_minMagnitude
            OneDayModelPostProcess.MaxDepth = one_day_model_post_process_maxDepth
            
            

    #--------------------------------------------------------------------------------------
    #
    # Test post-processing of the catalog data that is used as input for
    # one-day forecasts generation.
    #
    def testOneDayModelInputPostProcessing (self):
        """ Test post-processing of GeoNetNZ catalog for one-day \
forecast generation, and succeed."""

        # Setup test name
        CSEPTestCase.setTestName(self, 
                                 "GeoNetNZ-OneDayModelInputPostProcessing")

        # Invoke post-processing test for catalog that is used as input for one day forecast
        # generation
    
        # There are no uncertainties applied to the catalog
        self.__postProcessing(OneDayModelInputPostProcess.Type,
                              datetime.datetime(2008, 9, 8))


    #--------------------------------------------------------------------------------------
    #
    # Invoke post-processing of the catalog data used for the forecast 
    # evaluation tests, and evaluate generated catalog uncertainties if any
    #
    # Inputs: 
    #            post_process_type - Keyword identifying PostProcessing to be
    #                                          applied to the catalog data.
    #            test_date - Test date for post-processing.
    #            uncertainties_dir - Directory with reference catalogs with 
    #                                       uncertainties (used for evaluation of result data). 
    #                                       Default is None.
    #            pre_processed_file - Pre-processed data file for the test. 
    #                                 Default is CatalogDataSource.PreProcessedFile.
    #            args - Optioinal input arguments for post-processing object.
    #                   Default is None.
    #            skip_columns - Columns to skip during result validation. Default
    #                           is an empty list.
    #
    def __postProcessing(self,
                         post_process_type,
                         test_date,                           
                         uncertainties_dir = None,
                         pre_processed_file = CatalogDataSource.PreProcessedFile,
                         args = None,
                         skip_columns = []):
        """ Invoke catalog post-processing that is specific to the forecast class."""

  
        # Create post-processing object for the catalog data
        post_process = PostProcessFactory().object(post_process_type,
                                                   args)
        
        # Catalog filename used by post-processing
        matlab_catalog_file = post_process.files.catalog

        data_source_settings = {'start_date' : datetime.datetime(2006, 1, 1)}
        self.setDataSource(GeoNetNZDataSource.GeoNetNZDataSource.Type,
                           data_source_settings)

        GeographicalRegions.Region().set('NewZealand')

        # Instantiating a catalog object will take care
        # of generating a test directory
        catalog = RELMCatalog(CSEPTestCase.TestDirPath, 
                              DataSourceFactory().object(GeoNetNZDataSource.GeoNetNZDataSource.Type,
                                                         isObjReference=True), 
                              post_process)
        
        # Copy reference raw catalog data file to the test directory
        shutil.copyfile(os.path.join(CatalogPostProcessingTest.__referenceDataDir, 
                                     pre_processed_file),
                        os.path.join(CSEPTestCase.TestDirPath, 
                                     CatalogDataSource.PreProcessedFile))    


        # Copy original random numbers files to the test directory (used for generating
        # catalog uncertainties)
        if uncertainties_dir != None:
              random_dir = "%s/random" %(uncertainties_dir)
              shutil.copytree(os.path.join(CatalogPostProcessingTest.__referenceDataDir, 
                                           random_dir),
                              os.path.join(CSEPTestCase.TestDirPath, 
                                           "uncertainties"))
        
        
        # Create catalog and apply uncertainties to it. 
        catalog.create(test_date)
        
        ### Compare newly created catalog to the reference in ascii format

        # Define filenames and paths for reference and test data
        ascii_catalog_file = matlab_catalog_file.replace('.mat', '.dat')

        reference_catalog_file = "%s.%s" %(post_process_type, 
                                              ascii_catalog_file)
        
        reference_file = os.path.join(CatalogPostProcessingTest.__referenceDataDir, 
                                      reference_catalog_file)
        test_file = os.path.join(CSEPTestCase.TestDirPath, 
                                 ascii_catalog_file)

        
        CSEPLogging.getLogger(CatalogPostProcessingTest.__name__).info(
           "Comparing reference catalog file %s with generated catalog file %s..." \
           %(reference_file, test_file))

        error_msg = "Catalog %s post-processing failed." %(post_process_type)
        
        self.failIf(CSEPFile.compare(reference_file, test_file,
                                     skip_column_index = skip_columns) is False, 
                    error_msg)
        

        ### Compare all generated catalogs with uncertainties if any
        if uncertainties_dir is not None:
              error = "Uncertainty catalog %s post-processing failed "  \
                      %(post_process_type)
             
              num_catalogs = CSEP.Catalog.NumUncertainties + 1
              
              for index in xrange(1, num_catalogs):
                  # Compare generated catalog with uncertainty to the original data
                  filename = "catalog.uncert.%s.dat" %(index)
                  reference_file = os.path.join(CatalogPostProcessingTest.__referenceDataDir, 
                                                uncertainties_dir, 
                                                filename)
                  test_file = os.path.join(CSEPTestCase.TestDirPath, 
                                           "uncertainties", 
                                           filename)
                 
              
                  CSEPLogging.getLogger(CatalogPostProcessingTest.__name__).info(
                     "Comparing reference catalog file %s with generated catalog \
file %s..." %(reference_file, test_file))          
                 
                  error_msg = "%s (file index %s)" %(error, index)
                  self.failIf(CSEPFile.compare(reference_file, 
                                               test_file,
                                               precision=1E-6) is False, 
                              error_msg)    
                  
                  
# Invoke the module
if __name__ == '__main__':

   GeographicalRegions.Region().set('NewZealand')
   
   # Invoke all tests
   unittest.main()
        
# end of main
