"""
Module PublishResults
"""

__version__ = "$Revision: 3690 $"
__revision__ = "$Id: PublishResults.py 3690 2012-03-21 22:12:16Z liukis $"


import sys, os, unittest, shutil, glob, datetime

import CSEPFile, Environment
from CSEPTestCase import CSEPTestCase
from Dispatcher import Dispatcher
from DispatcherInitFile import DispatcherInitFile
from CSEPOptions import CommandLineOptions
from ANSSDataSource import ANSSDataSource


 #--------------------------------------------------------------------
 #
 # Validate that Dispatcher class is working properly.
 #
class PublishResults (CSEPTestCase):

   # Static data of the class
   
   # Unit tests use sub-directory of global reference data directory
   __referenceDataDir = os.path.join(CSEPTestCase.ReferenceDataDir, 
                                     'unitTest', 'publishResults')

   
   #--------------------------------------------------------------------
   #
   # This test verifies that Dispatcher publishes result plots to the
   # specified host and directory.
   #
   def testDispatcherPublishing(self):
      """ Confirm that Dispatcher publishes results plots to remote host \
('localhost' with passwordless connection is used in the test)."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())
      

      # Directory on remote machine to push results plots to
      publish_dir = os.path.join(CSEPTestCase.TestDirPath, "publish_plots")
      if os.path.exists(publish_dir) == False:
         os.mkdir(publish_dir)
         
      # Copy forecast group directory to the runtime test directory
      group_dir = "forecasts"
      shutil.copytree(os.path.join(PublishResults.__referenceDataDir, group_dir),
                      os.path.join(CSEPTestCase.TestDirPath, group_dir))
      
      # Copy dispatcher initialization file and replace forecast group 
      # directory with runtime path - it can't be a relative
      # path to the dispatcher's directory
      xmldoc = DispatcherInitFile(os.path.join(PublishResults.__referenceDataDir,
                                               "dispatcher_publish_results.init.xml"))
      groups = xmldoc.elements(DispatcherInitFile.ForecastGroupElement)
      groups[0].text = os.path.join(CSEPTestCase.TestDirPath, group_dir)

      # Use pre-processed data from specified directory
      dirs = xmldoc.elements(DispatcherInitFile.RootDirectoryElement)
      dirs[0].attrib[DispatcherInitFile.PreProcessedDataDirAttribute] = CSEPTestCase.ReferenceDataDir
      dirs[0].attrib[DispatcherInitFile.RawDataDirAttribute] = CSEPTestCase.ReferenceDataDir
      
      # Write modified file to the test directory
      init_file = os.path.join(CSEPTestCase.TestDirPath, "dispatcher.init.xml")
      
      fhandle = CSEPFile.openFile(init_file, 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      cwd = os.getcwd() 
      os.chdir(CSEPTestCase.TestDirPath)
      
      try:
         
         # Clear exceptions generated by other unit tests
         sys.exc_clear()
         del sys.argv[1:]
         
         # Simulate command-line arguments
         option = "%s=%s" %(CommandLineOptions.YEAR, CSEPTestCase.Date.year)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.MONTH, CSEPTestCase.Date.month)  
         sys.argv.append(option)         
         
         option = "%s=%s" %(CommandLineOptions.DAY, CSEPTestCase.Date.day)        
         sys.argv.append(option)         
         
         # Don't download raw data and don't pre-process
         sys.argv.append(CommandLineOptions.DOWNLOAD_RAW)
         sys.argv.append(CommandLineOptions.PREPROCESS_RAW)         
         
         # Enable forecast map generation
         sys.argv.append(CommandLineOptions.FORECAST_MAP)
         
         # Enable forecast master XML template 
         sys.argv.append(CommandLineOptions.FORECAST_TEMPLATE)
         
         option = "%s=0" %CommandLineOptions.WAITING_PERIOD
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_TEST_SIMULATIONS, 4)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_CATALOG_VARIATIONS, 4)
         sys.argv.append(option)

         # Use localhost as remote machine
         option = "%s=localhost" %(CommandLineOptions.PUBLISH_SERVER)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.PUBLISH_DIR, publish_dir)
         sys.argv.append(option)
                  
                  
         object = Dispatcher()
         object.run()
         
      finally:
            os.chdir(cwd)
            
    
      # Verify that forecast group configuration file has been published
      reference_file = os.path.join(publish_dir, "forecasts", "forecast.init.xml")
      error_msg = "Failed to publish %s forecast group configuration file" \
                  %reference_file
      self.failIf(os.path.exists(reference_file) == False, 
                  error_msg)

      #=========================================================================
      # Trac ticket #74: Post catalogs and scaled down forecasts used by 
      # evaluation tests
      #=========================================================================
      # Verify that observations for the forecast group have been published
      reference_file = os.path.join(publish_dir, 
                                    'forecasts',
                                    'catalogs', 
                                    '2006-11-02')
      error_msg = "Failed to publish %s observations for the forecast group" \
                  %reference_file
      self.failIf(os.path.exists(reference_file) == False, 
                  error_msg)
      
      catalog_files = glob.glob('%s/*catalog.nodecl.*' %reference_file)
      self.failIf(len(catalog_files) == 0,
                  "Missing observation catalogs in %s publishing directory" %reference_file)
      
      # Make sure that forecasts files are published
      forecasts_files = glob.glob(os.path.join(publish_dir,
                                               'forecasts',
                                               'ebel.mainshock*'))
      self.failIf(len(forecasts_files) == 0,
                  "Missing forecasts files of ebel.mainshock* pattern in %s publishing directory"
                  %publish_dir)
      #=========================================================================
      # End of Trac ticket #74
      #=========================================================================
            
      # Verify that plot files were published to the server
      reference_dir = os.path.join(publish_dir, "forecasts", "results",
                                    "%s" %CSEPTestCase.Date.date())      
      error_msg = "Failed to publish test results to the %s directory" \
                  %reference_dir
      self.failIf(os.path.exists(reference_dir) == False, 
                  error_msg)
      
      command = "%s/*svg*" %reference_dir
      svg_files = glob.glob(command)
      
      error_msg = "No SVG files found for daily test results under %s directory" \
                  %reference_dir
      self.failIf(len(svg_files) == 0, error_msg)

      command = "%s/*png*" %reference_dir
      png_files = glob.glob(command)
      
      error_msg = "No PNG map files found for daily test results under %s directory" \
                  %reference_dir
      self.failIf(len(png_files) == 0, error_msg)
      

   #-----------------------------------------------------------------------------
   #
   # This test verifies that Dispatcher does not attempt to publish result plots
   # if no such plot files exist.
   #
   def testNoPlotFiles(self):
      """ Confirm that Dispatcher does not attempt to publish results plots \
to remote host if no such plot files were generated by the run."""


      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Directory on remove machine to push results plots to
      publish_dir = os.path.join(CSEPTestCase.TestDirPath, "publish_plots")
      if os.path.exists(publish_dir) == False:
         os.mkdir(publish_dir)
         
      # Copy forecast group directory to the runtime test directory
      group_dir = "noForecasts"
      shutil.copytree(os.path.join(PublishResults.__referenceDataDir, group_dir),
                      os.path.join(CSEPTestCase.TestDirPath, group_dir))
      
      # Copy dispatcher initialization file and replace forecast group 
      # directory with runtime path - it can't be a relative
      # path to the dispatcher's directory
      xmldoc = DispatcherInitFile(os.path.join(PublishResults.__referenceDataDir,
                                               "dispatcher_dont_publish_results.init.xml"))
      groups = xmldoc.elements(DispatcherInitFile.ForecastGroupElement)
      groups[0].text = os.path.join(CSEPTestCase.TestDirPath, group_dir)

      # Use raw data from specified directory
      dirs = xmldoc.elements(DispatcherInitFile.RootDirectoryElement)
      dirs[0].attrib[DispatcherInitFile.PreProcessedDataDirAttribute] = CSEPTestCase.ReferenceDataDir
      dirs[0].attrib[DispatcherInitFile.RawDataDirAttribute] = CSEPTestCase.ReferenceDataDir
            
      # Write modified file to the test directory
      init_file = os.path.join(CSEPTestCase.TestDirPath, "dispatcher.init.xml")
      
      fhandle = CSEPFile.openFile(init_file, 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      cwd = os.getcwd() 
      os.chdir(CSEPTestCase.TestDirPath)
      
      try:
         
         # Clear exceptions generated by other unit tests         
         sys.exc_clear()         
         del sys.argv[1:]
         
         # Simulate command-line arguments
         option = "%s=%s" %(CommandLineOptions.YEAR, CSEPTestCase.Date.year)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.MONTH, CSEPTestCase.Date.month)  
         sys.argv.append(option)         
         
         option = "%s=%s" %(CommandLineOptions.DAY, CSEPTestCase.Date.day)        
         sys.argv.append(option)         
         
         # Don't download raw data and don't pre-process
         sys.argv.append(CommandLineOptions.DOWNLOAD_RAW)
         sys.argv.append(CommandLineOptions.PREPROCESS_RAW)
         
         option = "%s=0" %CommandLineOptions.WAITING_PERIOD
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_TEST_SIMULATIONS, 4)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_CATALOG_VARIATIONS, 4)
         sys.argv.append(option)

         # Use localhost as remote machine
         option = "%s=localhost" %(CommandLineOptions.PUBLISH_SERVER)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.PUBLISH_DIR, publish_dir)
         sys.argv.append(option)
                  
                  
         object = Dispatcher()
         object.run()
         
      finally:
            os.chdir(cwd)
            
            
      # Verify that plot files were published to the server
      reference_dir = os.path.join(publish_dir, "noForecasts", "results",
                                    "%s" %CSEPTestCase.Date.date())      
      error_msg = "Publish directory for non-existing test results should not exist: %s" \
                  %reference_dir
      self.failIf(os.path.exists(reference_dir) is True, 
                  error_msg)


   #----------------------------------------------------------------------------
   #
   # This test verifies that Dispatcher runtime directory and log file gets 
   # published
   #
   def testDispatcherRuntimeInfo(self):
      """ Confirm that Dispatcher publishes it's runtime directory and log file \
to remote host even if no result plot files were generated by the run."""


      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())

      # Directory on remove machine to push results plots to
      publish_dir = os.path.join(CSEPTestCase.TestDirPath, "publish_dispatcher_info")
      if os.path.exists(publish_dir) == False:
         os.mkdir(publish_dir)
         
      log_file = "dispatcher_log.txt"
      log_path = os.path.join(CSEPTestCase.TestDirPath, log_file)
         
      # Copy forecast group directory to the runtime test directory
      group_dir = "noForecasts"
      shutil.copytree(os.path.join(PublishResults.__referenceDataDir, group_dir),
                      os.path.join(CSEPTestCase.TestDirPath, group_dir))
      
      # Copy dispatcher initialization file and replace forecast group 
      # directory with runtime path - it can't be a relative
      # path to the dispatcher's directory
      xmldoc = DispatcherInitFile(os.path.join(PublishResults.__referenceDataDir,
                                               "dispatcher_dont_publish_results.init.xml"))
      groups = xmldoc.elements(DispatcherInitFile.ForecastGroupElement)
      groups[0].text = os.path.join(CSEPTestCase.TestDirPath, group_dir)

      # Use raw data from specified directory
      dirs = xmldoc.elements(DispatcherInitFile.RootDirectoryElement)
      dirs[0].attrib[DispatcherInitFile.PreProcessedDataDirAttribute] = CSEPTestCase.ReferenceDataDir
      dirs[0].attrib[DispatcherInitFile.RawDataDirAttribute] = CSEPTestCase.ReferenceDataDir
            
      # Write modified file to the test directory
      init_file = os.path.join(CSEPTestCase.TestDirPath, "dispatcher.init.xml")
      
      fhandle = CSEPFile.openFile(init_file, 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      cwd = os.getcwd() 
      os.chdir(CSEPTestCase.TestDirPath)
      
      try:
         
         # Clear exceptions generated by other unit tests         
         sys.exc_clear()         
         del sys.argv[1:]
         
         # Simulate command-line arguments
         option = "%s=%s" %(CommandLineOptions.YEAR, CSEPTestCase.Date.year)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.MONTH, CSEPTestCase.Date.month)  
         sys.argv.append(option)         
         
         option = "%s=%s" %(CommandLineOptions.DAY, CSEPTestCase.Date.day)        
         sys.argv.append(option)         
         
         # Don't download raw data and don't pre-process
         sys.argv.append(CommandLineOptions.DOWNLOAD_RAW)
         sys.argv.append(CommandLineOptions.PREPROCESS_RAW)
         
         option = "%s=0" %CommandLineOptions.WAITING_PERIOD
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_TEST_SIMULATIONS, 4)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_CATALOG_VARIATIONS, 4)
         sys.argv.append(option)

         # Use localhost as remote machine
         option = "%s=localhost" %(CommandLineOptions.PUBLISH_SERVER)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.PUBLISH_DIR, publish_dir)
         sys.argv.append(option)
                  
         # Use localhost as remote machine
         option = "%s=localhost" %(CommandLineOptions.PUBLISH_RUNTIME_SERVER)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.PUBLISH_RUNTIME_DIR, publish_dir)
         sys.argv.append(option)
         
         option = '%s=%s' %(CommandLineOptions.LOG_FILE,
                            log_path)
         sys.argv.append(option)
         
         # Just "touch" log file to guarantee it's existence 
         Environment.invokeCommand("touch %s" %log_path)
                  
         object = Dispatcher()
         object.run()
         
      finally:
            os.chdir(cwd)
            
            
      # Verify that Dispatcher's runtime directory is published
      dir_pattern = os.path.join(publish_dir, 
                                 'pid*/scec.csep*')
      
      dir_content = glob.glob(dir_pattern)
      
      # Metadata for raw catalog is created when SVN approach is used
      expected_num_files = 5 
      if ANSSDataSource().SVN.isWorkingCopy is False:
          expected_num_files = 4
      
      self.failIf(len(dir_content) != expected_num_files,
                  'Published runtime directory %s does not include expected \
number of scec.csep* files' %dir_pattern)
                              
      # Verify that Dispatcher's log file is published
      self.failIf(os.path.exists(os.path.join(publish_dir, log_file)) is False,
                  "Failed to publish Dispatcher's log file %s under %s directory" \
                  %(log_file, 
                    publish_dir))


   #--------------------------------------------------------------------
   #
   # This test verifies that Dispatcher publishes result plots to the
   # specified host and directory.
   #
   def testExistingModelInputCatalog(self):
      """ Confirm that Dispatcher publishes generated forecasts to remote host \
('localhost' with passwordless connection is used in the test). Pre-generated \
input catalog is used to generate forecasts files."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())
      

      # Directory on remote machine to push results plots to
      publish_dir = os.path.join(CSEPTestCase.TestDirPath, "publish_plots")
      if os.path.exists(publish_dir) == False:
         os.mkdir(publish_dir)
         
      # Copy forecast group directory to the runtime test directory
      group_dir = "oneDayForecasts"
      shutil.copytree(os.path.join(PublishResults.__referenceDataDir, group_dir),
                      os.path.join(CSEPTestCase.TestDirPath, group_dir))
      
      # Copy dispatcher initialization file and replace forecast group 
      # directory with runtime path - it can't be a relative
      # path to the dispatcher's directory
      xmldoc = DispatcherInitFile(os.path.join(PublishResults.__referenceDataDir,
                                               "dispatcher_existing_inputCatalog.init.xml"))
      groups = xmldoc.elements(DispatcherInitFile.ForecastGroupElement)
      groups[0].text = os.path.join(CSEPTestCase.TestDirPath, group_dir)

      # Use pre-processed data from specified directory
      dirs = xmldoc.elements(DispatcherInitFile.RootDirectoryElement)
      dirs[0].attrib[DispatcherInitFile.PreProcessedDataDirAttribute] = CSEPTestCase.ReferenceDataDir
      dirs[0].attrib[DispatcherInitFile.RawDataDirAttribute] = CSEPTestCase.ReferenceDataDir
      dirs[0].attrib[DispatcherInitFile.InputCatalogAttribute] = os.path.join(PublishResults.__referenceDataDir)
      
      # Write modified file to the test directory
      init_file = os.path.join(CSEPTestCase.TestDirPath, "dispatcher.init.xml")
      
      fhandle = CSEPFile.openFile(init_file, 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      cwd = os.getcwd() 
      os.chdir(CSEPTestCase.TestDirPath)
      
      test_date = datetime.datetime(2011, 9, 29)
      
      try:
         
         # Clear exceptions generated by other unit tests
         sys.exc_clear()
         del sys.argv[1:]
         
         # Simulate command-line arguments
         option = "%s=%s" %(CommandLineOptions.YEAR, test_date.year)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.MONTH, test_date.month)  
         sys.argv.append(option)         
         
         option = "%s=%s" %(CommandLineOptions.DAY, test_date.day)        
         sys.argv.append(option)         
         
         # Don't download raw data and don't pre-process
         sys.argv.append(CommandLineOptions.DOWNLOAD_RAW)
         sys.argv.append(CommandLineOptions.PREPROCESS_RAW)         
         
         option = "%s=0" %CommandLineOptions.WAITING_PERIOD
         sys.argv.append(option)
         
         object = Dispatcher()
         object.run()
         
      finally:
            os.chdir(cwd)
            
    
      # Make sure that forecasts files are generated
      forecasts_files = glob.glob(os.path.join(CSEPTestCase.TestDirPath,
                                               group_dir,
                                               'archive',
                                               '2011_9',
                                               'BogusForecastModel*'))
      self.failIf(len(forecasts_files) == 0,
                  "Missing forecasts files of BogusForecastModel* pattern in %s group directory"
                  %publish_dir)

      

# Invoke the module
if __name__ == '__main__':
   
   # Invoke all tests
   unittest.main()
        
# end of main
