"""
Module StageRawDataTest
"""

__version__ = "$Revision: 1922 $"
__revision__ = "$Id: PublishResults.py 1922 2009-02-18 06:02:45Z liukis $"


import sys, os, unittest, shutil, glob, datetime, re

import CSEPFile
from CSEPTestCase import CSEPTestCase
from Dispatcher import Dispatcher
from DispatcherInitFile import DispatcherInitFile
from CSEPPropertyFile import CSEPPropertyFile, TypeField
from CSEPOptions import CommandLineOptions
from ANSSDataSource import ANSSDataSource
from CatalogDataSource import CatalogDataSource
from OneDayModelInputPostProcess import OneDayModelInputPostProcess


 #--------------------------------------------------------------------
 #
 # Validate that Dispatcher class is working properly.
 #
class StageRawDataTest (CSEPTestCase):

   # Static data of the class
   
   # Unit tests use sub-directory of global reference data directory
   __referenceDataDir = os.path.join(CSEPTestCase.ReferenceDataDir, 
                                     'unitTest', 'stageRawData')

   
   #----------------------------------------------------------------------------
   #
   # This test verifies that Dispatcher can stage existing raw catalog data 
   # based on metadata for existing observation catalog. 
   #
   def testStageByCatalog(self):
      """ Confirm that Dispatcher stages existing raw catalog data based on \
metadata for existing observation catalog file ."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())
      

      # Copy forecast group directory to the runtime test directory
      group_dir = "RELM-mainshock-models"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, group_dir),
                      os.path.join(CSEPTestCase.TestDirPath, group_dir))

      # Copy dispatcher directory to the runtime test directory
      dispatcher_dir = "dispatcher"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, dispatcher_dir),
                      os.path.join(CSEPTestCase.TestDirPath, dispatcher_dir))
      
      # Copy dispatcher initialization file and replace forecast group 
      # directory with runtime path - it can't be a relative
      # path to the dispatcher's directory
      xmldoc = DispatcherInitFile(os.path.join(StageRawDataTest.__referenceDataDir,
                                               "dispatcher.init.xml"))
      groups = xmldoc.elements(DispatcherInitFile.ForecastGroupElement)
      groups[0].text = os.path.join(CSEPTestCase.TestDirPath, group_dir)

      # Write modified file to the test directory
      init_file = os.path.join(CSEPTestCase.TestDirPath, "dispatcher.init.xml")
      
      fhandle = CSEPFile.openFile(init_file, 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      cwd = os.getcwd() 
      os.chdir(CSEPTestCase.TestDirPath)
      
      try:
          
         # Replace directory paths with run-time directory of the test case
         self.__replaceDirWithRuntimeDir('dispatcher/runs/csep/2009_02/20090203082038',
                                         'dispatcher/')
    
         self.__replaceDirWithRuntimeDir('RELM-mainshock-models/catalogs/2009-01-03',
                                         'RELM-mainshock-models/')
         
         # Clear exceptions generated by other unit tests
         sys.exc_clear()
         del sys.argv[1:]

         test_date = datetime.datetime(2009, 1, 3)
         # Simulate command-line arguments
         option = "%s=%s" %(CommandLineOptions.YEAR, test_date.year)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.MONTH, test_date.month)  
         sys.argv.append(option)         
         
         option = "%s=%s" %(CommandLineOptions.DAY, test_date.day)        
         sys.argv.append(option)         
         
         # Don't download raw data and don't pre-process
         sys.argv.append(CommandLineOptions.DOWNLOAD_RAW)
         
         # Don't stage existing data products (forecasts, observations)
         sys.argv.append(CommandLineOptions.STAGING)
         
         # Enable forecast map generation
         sys.argv.append(CommandLineOptions.FORECAST_MAP)
         
         # Enable forecast master XML template 
         sys.argv.append(CommandLineOptions.FORECAST_TEMPLATE)
         
         option = "%s=0" %CommandLineOptions.WAITING_PERIOD
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_TEST_SIMULATIONS, 1)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_CATALOG_VARIATIONS, 1)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_DECLUSTER_SIMULATIONS, 1)
         sys.argv.append(option)

         object = Dispatcher()
         runtime_dir, error_code = object.run()

         # Verify that runtime directory for the test contains a copy of 
         # reference raw catalog data
         meta_files = glob.glob('%s/*%s' %(runtime_dir,
                                           CSEPPropertyFile.Metadata.Extension))
         found_raw_data = False
         
         for each_meta in meta_files:
             meta_obj = CSEPPropertyFile.Metadata(each_meta)
             meta_path, meta_file = os.path.split(meta_obj.originalDataFilename)
             if meta_file == 'import_raw.dat':
                 self.failIf(meta_obj.info[CSEPPropertyFile.Metadata.DataLinkKeyword] is None,
                             'Raw catalog file is not a link to existing file according to %s file' 
                             %meta_obj.file)
                 found_raw_data = True
             
         self.failIf(found_raw_data is False,
                     'Could not find metadata file corresponding to raw catalog data in run-time directory %s' 
                     %runtime_dir)
         
      finally:
            os.chdir(cwd)
            

   #----------------------------------------------------------------------------
   #
   # This test verifies that Dispatcher can stage existing raw catalog data 
   # based on metadata for existing observation catalog which is just a link to 
   # another catalog file.
   #
   def testStageByCatalogLink(self):
      """ Confirm that Dispatcher stages existing raw catalog data based on \
metadata for links to existing observation catalog file."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())
      

      # Copy forecast group directory to the runtime test directory
      reference_group_dir = "RELM-mainshock-models-withCatalogLinks"
      group_dir = "RELM-mainshock-models"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, 
                                   reference_group_dir),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   group_dir))

      # Copy dispatcher directory to the runtime test directory
      dispatcher_dir = "dispatcher"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, 
                                   dispatcher_dir),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   dispatcher_dir))
      
      # Copy dispatcher initialization file and replace forecast group 
      # directory with runtime path - it can't be a relative
      # path to the dispatcher's directory
      xmldoc = DispatcherInitFile(os.path.join(StageRawDataTest.__referenceDataDir,
                                               "dispatcher.init.xml"))
      groups = xmldoc.elements(DispatcherInitFile.ForecastGroupElement)
      groups[0].text = os.path.join(CSEPTestCase.TestDirPath, group_dir)

      # Write modified file to the test directory
      init_file = os.path.join(CSEPTestCase.TestDirPath, "dispatcher.init.xml")
      
      fhandle = CSEPFile.openFile(init_file, 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      cwd = os.getcwd() 
      os.chdir(CSEPTestCase.TestDirPath)
      
      try:
          
         # Replace directory paths with run-time directory of the test case
         self.__replaceDirWithRuntimeDir('dispatcher/runs/csep/2009_02/20090203082038',
                                         'dispatcher/')
    
         self.__replaceDirWithRuntimeDir('RELM-mainshock-models/catalogs/2009-01-03',
                                         'RELM-mainshock-models/')
         
         # Clear exceptions generated by other unit tests
         sys.exc_clear()
         del sys.argv[1:]

         test_date = datetime.datetime(2009, 1, 3)
         # Simulate command-line arguments
         option = "%s=%s" %(CommandLineOptions.YEAR, test_date.year)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.MONTH, test_date.month)  
         sys.argv.append(option)         
         
         option = "%s=%s" %(CommandLineOptions.DAY, test_date.day)        
         sys.argv.append(option)         
         
         # Don't download raw data and don't pre-process
         sys.argv.append(CommandLineOptions.DOWNLOAD_RAW)
         
         # Don't stage existing data products (forecasts, observations)
         sys.argv.append(CommandLineOptions.STAGING)
         
         # Enable forecast map generation
         sys.argv.append(CommandLineOptions.FORECAST_MAP)
         
         # Enable forecast master XML template 
         sys.argv.append(CommandLineOptions.FORECAST_TEMPLATE)
         
         option = "%s=0" %CommandLineOptions.WAITING_PERIOD
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_TEST_SIMULATIONS, 1)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_CATALOG_VARIATIONS, 1)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_DECLUSTER_SIMULATIONS, 1)
         sys.argv.append(option)

         object = Dispatcher()
         runtime_dir, error_code = object.run()

         # Verify that runtime directory for the test contains a copy of 
         # reference raw catalog data
         meta_files = glob.glob('%s/*%s' %(runtime_dir,
                                           CSEPPropertyFile.Metadata.Extension))
         found_raw_data = False
         
         for each_meta in meta_files:
             meta_obj = CSEPPropertyFile.Metadata(each_meta)
             meta_path, meta_file = os.path.split(meta_obj.originalDataFilename)
             if meta_file == CatalogDataSource._RawFile:
                 self.failIf(meta_obj.info[CSEPPropertyFile.Metadata.DataLinkKeyword] is None,
                             'Raw catalog file is not a link to existing file according to %s file' 
                             %meta_obj.file)
                 found_raw_data = True
             
         self.failIf(found_raw_data is False,
                     'Could not find metadata file corresponding to raw catalog data in run-time directory %s' 
                     %runtime_dir)
         
      finally:
            os.chdir(cwd)


   #----------------------------------------------------------------------------
   #
   # This test verifies that Dispatcher can stage existing raw catalog data 
   # based on metadata for existing observation catalog. Metadata represents 
   # SVN tag used to store raw data in SVN repository by previous processing. 
   #
   @unittest.skipIf(ANSSDataSource().SVN.isWorkingCopy is False,
                    "ANSS data source is not using SVN repository to store raw catalog data")
   def testStageByCatalogSVNTag(self):
      """ Confirm that Dispatcher stages existing raw catalog data based on \
SVN tag stored in metadata for existing observation catalog file ."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())
      

      # Copy forecast group directory to the runtime test directory
      group_dir = "RELM-mainshock-models"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, 
                                   group_dir),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   group_dir))

      # Copy dispatcher directory to the runtime test directory
      dispatcher_dir = "dispatcher"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, 
                                   dispatcher_dir),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   dispatcher_dir))
      
      # Copy dispatcher initialization file and replace forecast group 
      # directory with runtime path - it can't be a relative
      # path to the dispatcher's directory
      xmldoc = DispatcherInitFile(os.path.join(StageRawDataTest.__referenceDataDir,
                                               "dispatcher.init.xml"))
      groups = xmldoc.elements(DispatcherInitFile.ForecastGroupElement)
      groups[0].text = os.path.join(CSEPTestCase.TestDirPath, 
                                    group_dir)

      # Write modified file to the test directory
      init_file = os.path.join(CSEPTestCase.TestDirPath, 
                               "dispatcher.init.xml")
      
      fhandle = CSEPFile.openFile(init_file, 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      cwd = os.getcwd() 
      os.chdir(CSEPTestCase.TestDirPath)
      
      try:
          
         # Set SVN tag for raw data
         data_source = ANSSDataSource()
         svn_tag = datetime.datetime.now().strftime("%Y%m%d%H%M%S")
         data_source.SVN.setTag(svn_tag)
         svn_tag_url = data_source.SVN.tagURL()
         
         # Tag data in repository
         data_source.SVN.commit("%s data to test staging of raw data based on SVN tag, committed on %s" 
                                %(ANSSDataSource.Type,
                                  datetime.datetime.now()))
         
         # Replace directory paths with run-time directory of the test case
         self.__replaceDirWithRuntimeDir('dispatcher/runs/csep/2009_02/20090203082038',
                                         'dispatcher/',
                                         svn_tag_url) # Modify metadata of catalog to include SVN tag
    
         self.__replaceDirWithRuntimeDir('RELM-mainshock-models/catalogs/2009-01-03',
                                         'RELM-mainshock-models/')
         
          
         
         # Clear exceptions generated by other unit tests
         sys.exc_clear()
         del sys.argv[1:]

         test_date = datetime.datetime(2009, 1, 3)
         # Simulate command-line arguments
         option = "%s=%s" %(CommandLineOptions.YEAR, test_date.year)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.MONTH, test_date.month)  
         sys.argv.append(option)         
         
         option = "%s=%s" %(CommandLineOptions.DAY, test_date.day)        
         sys.argv.append(option)         
         
         # Don't download raw data and don't pre-process
         sys.argv.append(CommandLineOptions.DOWNLOAD_RAW)
         
         # Don't stage existing data products (forecasts, observations)
         sys.argv.append(CommandLineOptions.STAGING)
         
         # Enable forecast map generation
         sys.argv.append(CommandLineOptions.FORECAST_MAP)
         
         # Enable forecast master XML template 
         sys.argv.append(CommandLineOptions.FORECAST_TEMPLATE)
         
         option = "%s=0" %CommandLineOptions.WAITING_PERIOD
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_TEST_SIMULATIONS, 1)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_CATALOG_VARIATIONS, 1)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_DECLUSTER_SIMULATIONS, 1)
         sys.argv.append(option)

         object = Dispatcher()
         runtime_dir, error_code = object.run()

         # Verify that runtime directory for the test contains a copy of 
         # reference raw catalog data
         meta_files = glob.glob('%s/*%s' %(runtime_dir,
                                           CSEPPropertyFile.Metadata.Extension))
         found_raw_data = False
         
         for each_meta in meta_files:
             meta_obj = CSEPPropertyFile.Metadata(each_meta)
             meta_path, meta_file = os.path.split(meta_obj.originalDataFilename)
             if meta_file == CatalogDataSource._RawFile:
                 self.failIf(meta_obj.info[CSEPPropertyFile.Metadata.SVNKeyword] is None,
                             'Raw catalog file does not have SVN tag associated with it according to %s file' 
                             %meta_obj.file)
                 
                 self.failIf(meta_obj.info[CSEPPropertyFile.Metadata.SVNKeyword] != svn_tag_url,
                             'Raw catalog file should have %s SVN tag, found %s in %s file' 
                             %(svn_tag_url,
                               meta_obj.info[CSEPPropertyFile.Metadata.SVNKeyword],
                               meta_obj.file))

                 found_raw_data = True
             
         self.failIf(found_raw_data is False,
                     'Could not find metadata file corresponding to raw catalog data in run-time directory %s' 
                     %runtime_dir)
         
      finally:
            os.chdir(cwd)


   #----------------------------------------------------------------------------
   #
   # This test verifies that Dispatcher can stage existing raw catalog data 
   # based on metadata for existing observation catalog. Metadata represents 
   # SVN tag used to store raw data in SVN repository by previous processing. 
   #
   @unittest.skipIf(ANSSDataSource().SVN.isWorkingCopy is False,
                    "ANSS data source is not using SVN repository to store raw catalog data")
   def testStageByLinkToRawCatalogWithSVNTagByMetadata(self):
      """ Confirm that Dispatcher stages existing raw catalog data based on \
SVN tag stored in metadata for previously created observation catalog file. Metadata file \
resolves to runtime directory which stores link to other raw catalog that has been archived into \
SVN and only metadata file exists for raw catalog in original runtime directory."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())
      

      # Copy forecast group directory to the runtime test directory
      group_dir = "RELM-mainshock-models"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, 
                                   group_dir),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   group_dir))

      # Copy dispatcher directory to the runtime test directory
      dispatcher_dir = "dispatcher/"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, 
                                   'dispatcherWithLinks'),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   dispatcher_dir))
      
      # Copy dispatcher initialization file and replace forecast group 
      # directory with runtime path - it can't be a relative
      # path to the dispatcher's directory
      xmldoc = DispatcherInitFile(os.path.join(StageRawDataTest.__referenceDataDir,
                                               "dispatcher.init.xml"))
      groups = xmldoc.elements(DispatcherInitFile.ForecastGroupElement)
      groups[0].text = os.path.join(CSEPTestCase.TestDirPath, 
                                    group_dir)

      # Write modified file to the test directory
      init_file = os.path.join(CSEPTestCase.TestDirPath, 
                               "dispatcher.init.xml")
      
      fhandle = CSEPFile.openFile(init_file, 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      cwd = os.getcwd() 
      os.chdir(CSEPTestCase.TestDirPath)
      
      try:
          
         # Set SVN tag for raw data
         data_source = ANSSDataSource()
         svn_tag = datetime.datetime.now().strftime("%Y%m%d%H%M%S")
         data_source.SVN.setTag(svn_tag)
         svn_tag_url = data_source.SVN.tagURL()
         
         # Tag data in repository
         data_source.SVN.commit("%s data to test staging of raw data based on link to file that is stored in SVN, committed on %s" 
                                %(ANSSDataSource.Type,
                                  datetime.datetime.now()))
         
         # Replace directory paths with run-time directory of the test case
         self.__replaceDirWithRuntimeDir('dispatcher/runs/csep/2008_01/20080101082038',
                                         dispatcher_dir,
                                         svn_tag_url) # Modify metadata of raw catalog to include SVN tag
         
         self.__replaceDirWithRuntimeDir('dispatcher/runs/csep/2009_02/20090203082038',
                                         dispatcher_dir)
    
         self.__replaceDirWithRuntimeDir('RELM-mainshock-models/catalogs/2009-01-03',
                                         'RELM-mainshock-models/')
         
          
         
         # Clear exceptions generated by other unit tests
         sys.exc_clear()
         del sys.argv[1:]

         test_date = datetime.datetime(2009, 1, 3)
         # Simulate command-line arguments
         option = "%s=%s" %(CommandLineOptions.YEAR, test_date.year)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.MONTH, test_date.month)  
         sys.argv.append(option)         
         
         option = "%s=%s" %(CommandLineOptions.DAY, test_date.day)        
         sys.argv.append(option)         
         
         # Don't download raw data and don't pre-process
         sys.argv.append(CommandLineOptions.DOWNLOAD_RAW)
         
         # Don't stage existing data products (forecasts, observations)
         sys.argv.append(CommandLineOptions.STAGING)
         
         # Enable forecast map generation
         sys.argv.append(CommandLineOptions.FORECAST_MAP)
         
         # Enable forecast master XML template 
         sys.argv.append(CommandLineOptions.FORECAST_TEMPLATE)
         
         option = "%s=0" %CommandLineOptions.WAITING_PERIOD
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_TEST_SIMULATIONS, 1)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_CATALOG_VARIATIONS, 1)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_DECLUSTER_SIMULATIONS, 1)
         sys.argv.append(option)

         object = Dispatcher()
         runtime_dir, error_code = object.run()

         # Verify that runtime directory for the test contains a copy of 
         # reference raw catalog data
         meta_files = glob.glob('%s/*%s' %(runtime_dir,
                                           CSEPPropertyFile.Metadata.Extension))
         found_raw_data = False
         
         for each_meta in meta_files:
             meta_obj = CSEPPropertyFile.Metadata(each_meta)
             meta_path, meta_file = os.path.split(meta_obj.originalDataFilename)
             if meta_file == CatalogDataSource._RawFile:
                 self.failIf(meta_obj.info[CSEPPropertyFile.Metadata.SVNKeyword] is None,
                             'Raw catalog file does not have SVN tag associated with it according to %s file' 
                             %meta_obj.file)
                 
                 self.failIf(meta_obj.info[CSEPPropertyFile.Metadata.SVNKeyword] != svn_tag_url,
                             'Raw catalog file should have %s SVN tag, found %s in %s file' 
                             %(svn_tag_url,
                               meta_obj.info[CSEPPropertyFile.Metadata.SVNKeyword],
                               meta_obj.file))

                 found_raw_data = True
             
         self.failIf(found_raw_data is False,
                     'Could not find metadata file corresponding to raw catalog data in run-time directory %s' 
                     %runtime_dir)
         
      finally:
            os.chdir(cwd)


   #----------------------------------------------------------------------------
   #
   # This test verifies that Dispatcher can stage existing raw catalog data 
   # based on metadata for existing observation catalog. Metadata represents 
   # SVN tag used to store raw data in SVN repository by previous processing. 
   #
   @unittest.skipIf(ANSSDataSource().SVN.isWorkingCopy is False,
                    "ANSS data source is not using SVN repository to store raw catalog data")
   def testStageByLinkToRawCatalogWithSVNTag(self):
      """ Confirm that Dispatcher stages existing raw catalog data based on \
SVN tag stored in metadata for previously created observation catalog file. Metadata file \
resolves to runtime directory which stores link to other raw catalog that has been archived into \
SVN and only metadata file exists for raw catalog in original runtime directory."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())
      

      # Copy forecast group directory to the runtime test directory
      group_dir = "RELM-mainshock-models"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, 
                                   group_dir),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   group_dir))

      # Copy dispatcher directory to the runtime test directory
      dispatcher_dir = "dispatcher/"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, 
                                   'dispatcherWithLinksToExistingMeta'),
                      os.path.join(CSEPTestCase.TestDirPath, 
                                   dispatcher_dir))
      
      # Copy dispatcher initialization file and replace forecast group 
      # directory with runtime path - it can't be a relative
      # path to the dispatcher's directory
      xmldoc = DispatcherInitFile(os.path.join(StageRawDataTest.__referenceDataDir,
                                               "dispatcher.init.xml"))
      groups = xmldoc.elements(DispatcherInitFile.ForecastGroupElement)
      groups[0].text = os.path.join(CSEPTestCase.TestDirPath, 
                                    group_dir)

      # Write modified file to the test directory
      init_file = os.path.join(CSEPTestCase.TestDirPath, 
                               "dispatcher.init.xml")
      
      fhandle = CSEPFile.openFile(init_file, 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      cwd = os.getcwd() 
      os.chdir(CSEPTestCase.TestDirPath)
      
      try:
          
         # Set SVN tag for raw data
         data_source = ANSSDataSource()
         svn_tag = datetime.datetime.now().strftime("%Y%m%d%H%M%S")
         data_source.SVN.setTag(svn_tag)
         svn_tag_url = data_source.SVN.tagURL()
         
         # Tag data in repository
         data_source.SVN.commit("%s data to test staging of raw data based on link to file that is stored in SVN, committed on %s" 
                                %(ANSSDataSource.Type,
                                  datetime.datetime.now()))
         
         # Replace directory paths with run-time directory of the test case
         self.__replaceDirWithRuntimeDir('dispatcher/runs/csep/2008_01/20080101082038',
                                         dispatcher_dir,
                                         svn_tag_url) # Modify metadata of raw catalog to include SVN tag
         
         self.__replaceDirWithRuntimeDir('dispatcher/runs/csep/2009_02/20090203082038',
                                         dispatcher_dir)
    
         self.__replaceDirWithRuntimeDir('RELM-mainshock-models/catalogs/2009-01-03',
                                         'RELM-mainshock-models/')
         
          
         
         # Clear exceptions generated by other unit tests
         sys.exc_clear()
         del sys.argv[1:]

         test_date = datetime.datetime(2009, 1, 3)
         # Simulate command-line arguments
         option = "%s=%s" %(CommandLineOptions.YEAR, test_date.year)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.MONTH, test_date.month)  
         sys.argv.append(option)         
         
         option = "%s=%s" %(CommandLineOptions.DAY, test_date.day)        
         sys.argv.append(option)         
         
         # Don't download raw data and don't pre-process
         sys.argv.append(CommandLineOptions.DOWNLOAD_RAW)
         
         # Don't stage existing data products (forecasts, observations)
         sys.argv.append(CommandLineOptions.STAGING)
         
         # Enable forecast map generation
         sys.argv.append(CommandLineOptions.FORECAST_MAP)
         
         # Enable forecast master XML template 
         sys.argv.append(CommandLineOptions.FORECAST_TEMPLATE)
         
         option = "%s=0" %CommandLineOptions.WAITING_PERIOD
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_TEST_SIMULATIONS, 1)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_CATALOG_VARIATIONS, 1)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_DECLUSTER_SIMULATIONS, 1)
         sys.argv.append(option)

         object = Dispatcher()
         runtime_dir, error_code = object.run()

         # Verify that runtime directory for the test contains a copy of 
         # reference raw catalog data
         meta_files = glob.glob('%s/*%s' %(runtime_dir,
                                           CSEPPropertyFile.Metadata.Extension))
         found_raw_data = False
         
         for each_meta in meta_files:
             meta_obj = CSEPPropertyFile.Metadata(each_meta)
             meta_path, meta_file = os.path.split(meta_obj.originalDataFilename)
             if meta_file == CatalogDataSource._RawFile:
                 self.failIf(meta_obj.info[CSEPPropertyFile.Metadata.SVNKeyword] is None,
                             'Raw catalog file does not have SVN tag associated with it according to %s file' 
                             %meta_obj.file)
                 
                 self.failIf(meta_obj.info[CSEPPropertyFile.Metadata.SVNKeyword] != svn_tag_url,
                             'Raw catalog file should have %s SVN tag, found %s in %s file' 
                             %(svn_tag_url,
                               meta_obj.info[CSEPPropertyFile.Metadata.SVNKeyword],
                               meta_obj.file))

                 found_raw_data = True
             
         self.failIf(found_raw_data is False,
                     'Could not find metadata file corresponding to raw catalog data in run-time directory %s' 
                     %runtime_dir)
         
      finally:
            os.chdir(cwd)


   #----------------------------------------------------------------------------
   #
   # This test verifies that Dispatcher can stage existing raw catalog data 
   # based on metadata for existing forecast file. 
   #
   def testStageByForecast(self):
      """ Confirm that Dispatcher stages existing raw catalog data based on \
metadata for existing forecast file ."""

      # Setup test name
      CSEPTestCase.setTestName(self, 
                               self.id())
      

      # Copy forecast group directory to the runtime test directory
      group_dir = "one-day-bogus-models"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, group_dir),
                      os.path.join(CSEPTestCase.TestDirPath, group_dir))

      # Copy dispatcher directory to the runtime test directory
      dispatcher_dir = "dispatcher"
      shutil.copytree(os.path.join(StageRawDataTest.__referenceDataDir, dispatcher_dir),
                      os.path.join(CSEPTestCase.TestDirPath, dispatcher_dir))
      
      # Copy dispatcher initialization file and replace forecast group 
      # directory with runtime path - it can't be a relative
      # path to the dispatcher's directory
      xmldoc = DispatcherInitFile(os.path.join(StageRawDataTest.__referenceDataDir,
                                               "dispatcher-one-day-models.init.xml"))
      groups = xmldoc.elements(DispatcherInitFile.ForecastGroupElement)
      groups[0].text = os.path.join(CSEPTestCase.TestDirPath, group_dir)

      # Write modified file to the test directory
      init_file = os.path.join(CSEPTestCase.TestDirPath, "dispatcher.init.xml")
      
      fhandle = CSEPFile.openFile(init_file, 
                                  CSEPFile.Mode.WRITE)
      xmldoc.write(fhandle)
      fhandle.close()
      
      cwd = os.getcwd() 
      os.chdir(CSEPTestCase.TestDirPath)
      
      try:
          
         # Replace directory paths with run-time directory of the test case
         self.__replaceDirWithRuntimeDir('dispatcher/runs/liukis/2009_02/20090223140009',
                                         'dispatcher/')
    
         self.__replaceDirWithRuntimeDir('one-day-bogus-models/archive/2009_2',
                                         'one-day-bogus-models/')
         
         # Clear exceptions generated by other unit tests
         sys.exc_clear()
         del sys.argv[1:]

         test_date = datetime.datetime(2009, 2, 20)
         # Simulate command-line arguments
         option = "%s=%s" %(CommandLineOptions.YEAR, test_date.year)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.MONTH, test_date.month)  
         sys.argv.append(option)         
         
         option = "%s=%s" %(CommandLineOptions.DAY, test_date.day)        
         sys.argv.append(option)         
         
         # Don't download raw data and don't pre-process
         sys.argv.append(CommandLineOptions.DOWNLOAD_RAW)
         
         # Don't stage existing data products (forecasts, observations)
         sys.argv.append(CommandLineOptions.STAGING)
         
         option = "%s=0" %CommandLineOptions.WAITING_PERIOD
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_TEST_SIMULATIONS, 1)
         sys.argv.append(option)
         
         option = "%s=%s" %(CommandLineOptions.NUM_CATALOG_VARIATIONS, 1)
         sys.argv.append(option)
         
         object = Dispatcher()
         runtime_dir, error_msg = object.run()
         
         if error_msg is not None:
          self.fail("Unexpected error: %s" %error_msg)

         # Verify that runtime directory for the test contains a copy of 
         # reference raw catalog data
         meta_files = glob.glob('%s/*%s' %(runtime_dir,
                                           CSEPPropertyFile.Metadata.Extension))
         found_raw_data = False
         
         for each_meta in meta_files:
             meta_obj = CSEPPropertyFile.Metadata(each_meta)
             meta_path, meta_file = os.path.split(meta_obj.originalDataFilename)
             if meta_file == CatalogDataSource._RawFile:
                 self.failIf(meta_obj.info[CSEPPropertyFile.Metadata.DataLinkKeyword] is None,
                             'Raw catalog file is not a link to existing file according to %s file' 
                             %meta_obj.file)
                 found_raw_data = True
             
         self.failIf(found_raw_data is False,
                     'Could not find metadata file corresponding to raw catalog data in run-time directory %s' 
                     %runtime_dir)

         # Test for fix of Trac ticket #306: Don't preserve OneDayModelInputCatalog 
         # data products within testing framework
         found_input_catalog = False
         input_catalog_name = OneDayModelInputPostProcess().files.catalog
         
         for each_meta in meta_files:
             meta_obj = CSEPPropertyFile.Metadata(each_meta)
             meta_path, meta_file = os.path.split(meta_obj.originalDataFilename)
             if meta_file == input_catalog_name:
                 # Extract data filename from corresponding metadata filename
                 datafile = re.split(CSEPPropertyFile.Metadata.Extension, each_meta)[0]
                 self.failIf(os.path.exists(datafile),
                             'Input catalog file %s, that corresponds to %s file, should not exist' 
                             %(datafile,
                               meta_obj.file))
                 found_input_catalog = True
                 break
             
         self.failIf(found_input_catalog is False,
                     'Could not find metadata file corresponding to input catalog data in run-time directory %s' 
                     %runtime_dir)
          
      finally:
            os.chdir(cwd)
            
            
   #----------------------------------------------------------------------------
   #
   # This method creates metadata files under specified directory. Original
   # metadata files are located under 'dir_path/meta' sub-directory.
   # The method changes full paths to corresponding data file and runtime directory
   # for Dispatcher process that generated that data file with run-time directory 
   # of the test case. 
   #
   # Input: 
   #        dir_path - Path to the directory to create metadata files under.
   #        keyword - Keyword to split directory path by.
   #
   def __replaceDirWithRuntimeDir(self, 
                                  dir_path, 
                                  keyword,
                                  svn_tag = None):
      """ Overwrite paths for data file and runtime directory for Dispatcher
          process that generated it with run-time directory of test case."""

      
      meta_files = glob.glob('%s/meta/*%s' %(dir_path,
                                             CSEPPropertyFile.Metadata.Extension))
      
      __token_index = 0
      __value_index = 1 
      
      for each_file in meta_files:

          fhandle = CSEPFile.openFile(each_file)
          
          new_meta_filename = each_file.replace('/meta', '')

          new_fhandle = CSEPFile.openFile(new_meta_filename, 
                                          CSEPFile.Mode.WRITE)
         
          try:
   
             lines = fhandle.readlines()
             
             # Very first line is a commented out original file name for archived
             # file         
             new_fhandle.write(lines[0])
             
             for line in lines[1:]:
   
                tokens = [key.strip() for key in line.split(CSEPPropertyFile.Separator)]

                # Each line is in 'key=value' format, extract keyword from line of file
                line_keyword = tokens[__token_index]
                 
                # Extract data file information   
                if line_keyword in [CSEPPropertyFile.Metadata.DataFileKeyword, 
                                    CSEPPropertyFile.Metadata.DataLinkKeyword]:
                   
                   data_path, data_file = os.path.split(tokens[__value_index])
                   
                   if len(data_path) != 0:
                       
                       # Full path is provided, replace with run-time directory
                       # of test case
                       path_tokens = tokens[__value_index].split(keyword)
                       
                       tokens[__value_index] = os.path.join(CSEPTestCase.TestDirPath,
                                                            keyword, path_tokens[1])
                       
                       line = CSEPPropertyFile.Separator.join(tokens)
                       

                #---------------------------------------------------------------     
                # Extract dispatcher run-time directory from the line that captures
                # options used by Dispatcher instance
                # ProducedBy = ['/usr/local/cruise/projects/CSEP/checkout/src/generic/Dispatcher.py',
                #               '--year=2008', '--month=2', '--day=24', 
                #               '--configFile=/home/csep/operations/cronjobs/dispatcher_daily.init.xml', 
                #               '--waitingPeriod=31', '--disableMatlabDisplay', 
                #               '--enableForecastXMLTemplate', '--enableForecastMap',
                #               '--publishServer=csep-usc@intensity.usc.edu', 
                #               '--publishDirectory=/var/www/html/csep/data/us/usc', 
                #               '--logFile=/home/csep/operations/dispatcher/logs/daily_2008-2-24', 
                #               'runtimeDirectory=/home/csep/operations/dispatcher/runs/csep/20080224000501',
                #               'runtimeTestDate=2008-01-24'] = string
                elif line_keyword == CSEPPropertyFile.Metadata.ProducedByKeyword:
                   
                   # Join value of the token using the same separator 
                   # it was used to split it
                   options = CSEPPropertyFile.Separator.join(tokens[__value_index:])
                   
                   options_values = options.split(',')
                   
                   # Extract runtimeDirectory value
                   for pair_index, each_pair in enumerate(options_values):
                      
                      # If there key=value pair
                      if each_pair.find(CSEPPropertyFile.Separator) >= 0:
                         
                         # Get rid of single quotes
                         values = each_pair.split(CSEPPropertyFile.Separator)
                         
                         key = values[__token_index]
                         value = values[__value_index]
                         
                         if key.strip() == '\'' + CSEPPropertyFile.Metadata.RuntimeDirectoryKeyword:

                             path_tokens = value.strip().split('dispatcher/')
                           
                             value = os.path.join(CSEPTestCase.TestDirPath,
                                                  'dispatcher/', path_tokens[1])
                                                    
                             options_values[pair_index] = CSEPPropertyFile.Separator.join([key, value])
                             
                             line = CSEPPropertyFile.Separator.join([tokens[__token_index], 
                                                                     ','.join(options_values)])
                             break
               
                new_fhandle.write(line)
             
             if svn_tag is not None:
                 tokens = [CSEPPropertyFile.Metadata.SVNKeyword,
                           svn_tag,
                           TypeField.STRING]
                 line = CSEPPropertyFile.Separator.join(tokens)
                 new_fhandle.write(line)
                         
          finally:
              fhandle.close()              
              new_fhandle.close()          
      


# Invoke the module
if __name__ == '__main__':
   
   # Invoke all tests
   unittest.main()
        
# end of main
