"""
Module ETAS_DROneYearModel to be used as 30-minute model for comparison to ETAS_DR 30-minute model
"""

__version__ = "$Revision$"
__revision__ = "$Id$"


import os, datetime

import Environment, CSEPFile, CSEPLogging
from Forecast import Forecast
from OneDayForecast import OneDayForecast
from ThirtyMinutesForecast import ThirtyMinutesForecast
from EEPASV12_10Model import EEPASV12_10Model
from CatalogDataSource import CatalogDataSource


#--------------------------------------------------------------------------------
#
# ETAS_DR forecast model.
#
# This class is designed to invoke one-day PPE and ETAS forecast models. 
# It prepares input catalog data, formats control file with model parameters,
# and invokes the model. It places forecast file under user specified directory.
#
class ETAS_DROneDayModel (ThirtyMinutesForecast):

    # Static data of the class
    
    # Keyword identifying the type of the forecast
    Type = "ETAS_DR" + OneDayForecast.Type
    
    #----------------------------------------------------------------------------
    #
    # Initialization.
    #
    # Input: 
    #        dir_path - Directory to store forecast file to.
    #        args - Optional arguments for the model. EEPAS-ETAS models don't 
    #               require any configurable inputs by the CSEP system.
    #        param_file - Parameter file for a specific model configuration.
    #        use_weights - Flag if model shouuld use weights. Default is 0 (False).
    # 
    def __init__ (self, dir_path, args=None):
        """ Initialization for ETASModel class"""
        
        ThirtyMinutesForecast.__init__(self, dir_path)
        

    #----------------------------------------------------------------------------
    #
    # Return keyword identifying the model.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the type
    #
    def type (self):
        """ Returns keyword identifying the forecast model type."""
        
        return self.Type


    #----------------------------------------------------------------------------
    #
    # Return sub-type keyword identifying the model: based on testing region.
    #
    # Input: None.
    #
    # Output:
    #           String identifying the sub-type
    #
    def subtype (self):
        """ Returns keyword identifying the forecast model sub-type."""

        # Specify selected type of forecast as the sub-type for the model        
        return 'Md3'


    #----------------------------------------------------------------------------
    #
    # Write input parameter file for the model.
    #
    # Input: None
    #        
    def writeParameterFile (self, 
                            filename = None):
        """ Format input parameter file for the model.
            Created file will be used by Fortran executable that invokes the
            model."""

        fhandle = Forecast.writeParameterFile(self)
        
        
    #----------------------------------------------------------------------------
    #
    # Invoke the model.
    #
    # Input: None
    #        
    def run (self):
        """ Run ETAS_DR model."""
        
        raise "Should not invoke %s model" %self.Type


    #---------------------------------------------------------------------------
    #
    # Creates filename for PPE foreacast if it does not exist yet.
    #
    # Input:
    #        start_date - Start date of the forecast. Default is None.
    #
    # Output:
    #        Handle to the control file.
    def ppeForecastFile (self):
        """ Creates filename for PPE foreacast."""
        
        return self.filename().replace(self.type(), 
                                       self.type() + "PPE")
        
    #---------------------------------------------------------------------------
    #
    # Create forecast.
    # This method is overwritten to create metadata file per each forecast file
    # as generated by the model.
    #
    # Input: 
    #       test_date - datetime object that represents testing date.
    #       catalog_dir - Directory with catalog data
    #       archive_dir - Directory to store intermediate model results if any.
    #                     Default is None.    
    #        
    # Output:
    #        Name of created forecast file.
    #
    def create (self, 
                test_date, 
                catalog_dir, 
                archive_dir=None, 
                data_source = CatalogDataSource()):
        """ Generate forecast."""
        
        # Invoke the models
        result_files = Forecast.create(self,
                                       test_date, 
                                       catalog_dir, 
                                       archive_dir)

        # Check if PPE forecast file wasn't generated (ETAS forecast was staged) 
        # and PPE file is archived ---> stage the file
        file_is_staged = False
        if self.ppeForecastFile() is not None and \
           os.path.exists(os.path.join(self.dir,
                                       self.ppeForecastFile())) is False and \
           archive_dir is not None:
            
           file_is_staged = self.stage([self.ppeForecastFile()],
                                       os.path.join(archive_dir,
                                                    Forecast.archiveDir(self.ppeForecastFile())))


        if self.ppeForecastFile() is not None:
            result_files.append(os.path.basename(self.ppeForecastFile()))

        
        return result_files
