
"""
utility to create forecasts GZIP archives

"""

import os, glob

import CSEPFile
from Forecast import Forecast
from CSEPLogging import CSEPLogging


class ForecastsFiles:

   __dryRun = True
   
   def archive (self, 
                dir,       # top-level directory with existing forecasts
                dry_run):  # dry run: don't actually rename files 
     """ Create GZIP archives of existing forecasts, and remove 
         original forecasts files."""

     self.__dryRun = dry_run
     dir_path = os.path.join(dir,
                             'archive')
     
     entries = os.listdir(dir_path)
     cwd = os.getcwd()
     
     for each_dir in entries:
         each_dir_path = os.path.join(dir_path,
                                      each_dir)
         if os.path.isdir(each_dir_path) is True:
     
             self.__archiveFiles(each_dir_path)
                 
             os.chdir(cwd)


   def __archiveFiles (self, 
                       dir_path):  # directory with forecasts files
     """ Create GZIP archive of existing forecasts under specified directory."""

     CSEPLogging.getLogger(__name__).info("Processing directory: %s" %dir_path)
     
     os.chdir(dir_path)
             
     found_files = glob.glob('*.dat')

     for each_file in found_files:
         archive_filename = each_file + CSEPFile.Extension.TARGZ
         CSEPLogging.getLogger(__name__).info("Archiving %s to %s" %(each_file,
                                                                     archive_filename))
         
         if not self.__dryRun:
             archive = CSEPFile.GZIPArchive(archive_filename,
                                            CSEPFile.Mode.WRITE)
             archive.add(os.path.basename(each_file))
             del archive # Force file closure
                                
             # Create metadata file for the forecast 
             comment = "Archive of %s forecast file in %s format" \
                       %(each_file,
                         CSEPFile.Extension.toFormat(each_file))
                                
             Forecast.metadata(archive_filename,
                               comment)
             
             os.remove(each_file)


if __name__ == "__main__":
    
    import optparse
    
    
    command_options = optparse.OptionParser()
    
    command_options.add_option('--forecastDir',
                               dest='forecast_dir',
                               default=None,
                               help='Top-level forecasts directory')

    command_options.add_option('--disableDryRun',
                               dest='dry_run',
                               default=True,
                               action='store_false',
                               help='Invoke dry run of the program. Default is True.')
    
    
    (values, args) = command_options.parse_args()
    
    c = ForecastsFiles()
    c.archive(values.forecast_dir,
              values.dry_run)
    
