"""
Utility to restore medadata files for one-day forecasts based on information 
stored in run-time directory.
"""

__version__ = "$Revision$"
__revision__ = "$Id$"

import os, datetime, glob, sys, time
import CSEPFile, CSEP
from CSEPPropertyFile import CSEPPropertyFile


#--------------------------------------------------------------------------------
#
# RestoreMetadata
#
# This module is designed to restore forecasts metadata files based on the
# information files stored in Dispatcher's runtime directory.
#
class ForecastMetadata (object):

    # Static data members

    # Top-level directory with forecasts files 
    ForecastDir = "/home/csep/operations/SCEC-natural-laboratory/one-day-models/forecasts/archive"

    # Top-level directory with Dispatcher's runtime directories
    DispatcherDir = "/home/csep/operations/dispatcher/runs/csep"
    
    # Field separator
    Separator = "="
    
    # Filename separator
    NameSeparator ="."

    # Comment identifier
    Comment = "#"
   
    # Naming conventions for models for which metadata files are being restored
    __modelFiles = {'STEP' : '.mat',
                    'ETAS' : '.dat'}
     
    
    # Structure-like class that represents content of the metadata file
    class File (object):

       # Extension for metadata files.
       Extension = ".meta"
    
      
       # Keywords used by metadata file
       ProducedByKeyword = "ProducedBy"
       TestDateKeyword = 'runtimeTestDate='
       
     
       #----------------------------------------------------------------------------
       #
       # Initialization.
       #
       # Input: 
       #        filename - Name of the metadata file
       #
       def __init__ (self, filename):
          """Constructor for the MetadataFile class."""
         
          # Store name for metadata file
          self.file = filename
          
          self.testDate = None
          self.producedByArgs = []          

          
          fhandle = CSEPFile.openFile(self.file)
         
          try:
   
             lines = fhandle.readlines()         
   
   
             # Very first line is a comment with original data filename that 
             # corresponds to the metadata file
             for line in lines[1:]:
   
                # Use 'key=' as separator 
                tokens = [key.strip() for key in line.split(ForecastMetadata.File.TestDateKeyword)]
                
                if len(tokens) == 2:

                    # Each line is in 'key=value' format, extract value from tokens
                    values = tokens[1].split("'")
                    self.testDate = datetime.datetime.strptime(values[0], 
                                                               "%Y-%m-%d")
                    
                    # Extract list of arguments:
                    tokens = line.split('[')
                    args = tokens[1].split(']')[0]
                    self.producedByArgs = eval('[' + args + ']')
                    break
                
   
          finally:
            
             fhandle.close()


       #----------------------------------------------------------------------------
       #
       # Display object content.
       #
       # Input: None
       #
       def print_info (self):
          """Display extracted content for the Metadata object."""
         
          print "File ", self.file, ":", self.testDate, "\n"


    #===============================================================================
    # Constructor for RestoreMetadata class
    #
    # Inputs:
    #         dirpath - Directory path to model files.
    #
    #===============================================================================
    def __init__ (self, dirpath):
        """ Initialize ForecastMetadata class."""
   
        # Find one-day models within specified directory
        self.__forecasts = []

        # Forecast archive directory follows 'YYYY_M' format
        dir_path, dir_name = os.path.split(dirpath)
        dir_test_date = datetime.datetime.strptime(dir_name,  '%Y_%m')
         
        # Collect forecasts files as generated by the models in specified directory 
        for model_name, model_ext in ForecastMetadata.__modelFiles.iteritems():
            self.__forecasts.extend(glob.glob('%s/%s*%s%s' %(dirpath,
                                                             model_name,
                                                             dir_test_date.year,
                                                             model_ext)))
        
        
    #================================================================================
    #  restore
    # 
    # Restore metadata file for existing forecast files. Report if metadata file
    # already exists, and do nothing.
    #
    # Inputs: None
    #================================================================================
    def restore(self):
        """Restore metadata files for existing forecasts based on timestamp \
           for the file"""
         
        for each_forecast in self.__forecasts:

            forecast_metafile = each_forecast + CSEPPropertyFile.Metadata.Extension   

            # Don't create metadata file if it already exists 
            if os.path.exists(forecast_metafile) is True:
                print "Metadata file %s already exists" %forecast_metafile
                continue
            
            print 'Looking up runtime directories for', each_forecast
            
            # Detect creation time of the file
            forecast_file_time = datetime.datetime.strptime(ForecastMetadata.creationTime(each_forecast),
                                                            CSEP.Time.ISO8601Format)
            
            # Extract forecast start date from the filename
            forecast_path, forecast_name = os.path.split(each_forecast)
            name_tokens = CSEPFile.Name.extension(forecast_name).split('_')
            forecast_date = datetime.datetime.strptime('_'.join(name_tokens[1:]), 
                                                       '%m_%d_%Y')
            
            
            # Search for runtime dispatcher directories with OneDayModelInputCatalog.dat 
            # file stored in them on the day when forecast was generated
            glob_pattern = '%s/%s*/OneDayModelInputCatalog.dat' %(ForecastMetadata.DispatcherDir,
                                                                  forecast_file_time.strftime('%Y%m%d'))
            
            print 'Pattern:', glob_pattern
            
            catalog_dirs = glob.glob(glob_pattern)
            
            if len(catalog_dirs) == 0:
                print "ERROR: No Dispatcher directories with OneDayModelInputCatalog.dat are found for %s\n\n" %forecast_name
            else:

                found_dir = False
                
                for each_catalog_path in catalog_dirs:

                    each_dir, each_file = os.path.split(each_catalog_path)
                    print "Processing %s directory" %each_dir
            
                    # Locate metadata file for software version
                    sw_meta_file = glob.glob("%s/*SoftwareStatus*%s" %(each_dir,
                                                                       ForecastMetadata.File.Extension))
                    if len(sw_meta_file) == 0:
                        msg = "SoftwareStatus metadata file is missing in %s directory" \
                              %each_dir
                        raise RuntimeError, msg 
            
                    # Read file in and extract runtime test date which was processed
                    # by that instance of the Dispatcher
                    sw_meta = ForecastMetadata.File(sw_meta_file[0])
                    end_date = sw_meta.testDate + datetime.timedelta(days=1)
            
                    # Test date processed by the Dispatcher matches date of the forecast
                    if sw_meta.testDate.date() == forecast_date.date():
                        print "Identified runtime directory %s for %s" %(each_dir,
                                                                         forecast_name)
                        
                        forecast_metafile = each_forecast + CSEPPropertyFile.Metadata.Extension   

                        print "Creating metadata file", forecast_metafile, '\n\n'
                        
                        # Create the file
                        comment = "%s forecast file with start date '%s' and end date '%s' (metadata file is restored manually by %s)" \
                                  %(name_tokens[0], sw_meta.testDate, end_date, sys.argv)
                                 
                        # Extact command-line arguments from existing metadata file
                        CSEPPropertyFile.createMetafile(forecast_metafile, 
                                                        forecast_name,
                                                        CSEPFile.Extension.toFormat(forecast_name),
                                                        comment,
                                                        args_list = sw_meta.producedByArgs)
                        found_dir = True
                        break
                        
                if found_dir is False:
                   print "ERROR: could not identify runtime directory for ", each_forecast      


    #----------------------------------------------------------------------------
    #
    # Get creation time of the file.
    # This is a static method of the class.
    #
    # Input:
    #          filename - Name of the file.
    #
    # Output:
    #          String representation of ISO8601 format creation time for the file.
    #
    @staticmethod 
    def creationTime (filename):
        """ Get creation time of the file."""
        
        local_time = time.localtime(os.stat(filename).st_mtime)
        
        iso_8601 = time.strftime(CSEP.Time.ISO8601Format, 
                                 local_time)
        return iso_8601
    

    # End of Metadata class
   

# Invoke the module
if __name__ == '__main__':


    import optparse
    
    
    command_options = optparse.OptionParser()
    
    command_options.add_option('--forecastDir',
                               dest='forecast_dir',
                               default=None,
                               help="Directory with forecasts files")

    (values, args) = command_options.parse_args()

    meta = ForecastMetadata(values.forecast_dir)
    meta.restore()    
